# Dog Platformer Game - Comprehensive Strategy Plan

## Project Overview

**Project Name:** Dog Adventure Platformer
**Technology:** Three.js (2D Mode)
**Scope:** MVP/Prototype
**Target Platform:** Web (Desktop + Mobile)

### Vision
Create an engaging 2D platformer game featuring an animated dog character that can run, jump, and collect items. The game will use the 16-frame dog animation cycle to create smooth, appealing character movement.

---

## Asset Analysis

### Animation Frames Inventory
- **Total Frames:** 16 PNG files (dog_1.png through dog_16.png)
- **Style:** Cartoon dog with blue ears, white body, purple nose
- **Transparent Background:** Yes (PNG with alpha channel)

### Frame Breakdown by Animation State
Based on visual analysis:

| Frame Range | Animation State | Description |
|-------------|----------------|-------------|
| 1-2 | Idle/Sitting | Dog sitting front view, tail position varies |
| 3-8 | Run Cycle (Side) | Dog running animation, side view, legs in motion |
| 9-12 | Run Cycle (Alternate) | Continuation of run, different leg positions |
| 13-16 | Turn/Back View | Dog turning or back view positions |

### Animation Mapping Strategy
- **Idle Animation:** Frames 1-2 (loop)
- **Run Animation:** Frames 3-12 (loop)
- **Jump Animation:** Frame 2 (static or custom interpolation)
- **Land Animation:** Frame 1 (transition back to idle)
- **Turn Animation:** Frames 13-16 (for direction changes)

---

## Technical Architecture

### Technology Stack

#### Core Framework
- **Three.js** (2D orthographic camera mode)
  - Lightweight for 2D sprite rendering
  - WebGL acceleration for smooth performance
  - Easy texture/sprite management
  - Extensible to 3D if needed in future

#### Additional Libraries (Minimal)
- **No external physics engine initially** (custom gravity/collision for MVP)
- **Pure JavaScript** (ES6+)
- **HTML5 Canvas** (Three.js WebGL renderer)

### System Architecture

```
┌─────────────────────────────────────────────┐
│           Game Engine Core                  │
├─────────────────────────────────────────────┤
│  - Game Loop (requestAnimationFrame)        │
│  - State Manager (idle, running, jumping)   │
│  - Scene Manager (levels, transitions)      │
└─────────────────────────────────────────────┘
         │              │              │
    ┌────┴────┐    ┌────┴────┐   ┌────┴────┐
    │ Render  │    │ Physics │   │  Input  │
    │ System  │    │ System  │   │ Manager │
    └─────────┘    └─────────┘   └─────────┘
         │              │              │
    ┌────┴────────────┬─┴────────┬─────┴─────┐
    │                 │          │           │
┌───▼────┐    ┌──────▼───┐  ┌───▼────┐  ┌───▼────┐
│Sprite  │    │Collision │  │Keyboard│  │ Touch  │
│Manager │    │Detection │  │Handler │  │Handler │
└────────┘    └──────────┘  └────────┘  └────────┘
```

---

## Game Mechanics Design

### Core Mechanics (MVP)

#### 1. Player Movement
- **Horizontal Movement:** Left/Right arrow keys or A/D
- **Speed:** 200-300 pixels per second
- **Acceleration:** Smooth ramp-up (0.3s to max speed)
- **Deceleration:** Quick stop when key released

#### 2. Jumping System
- **Jump Trigger:** Spacebar or Up arrow or Tap (mobile)
- **Jump Height:** ~150-200 pixels
- **Jump Curve:** Parabolic (gravity-based)
- **Double Jump:** Not in MVP (can add later)
- **Coyote Time:** 0.1s grace period after leaving platform

#### 3. Gravity & Physics
- **Gravity Constant:** 980 pixels/s² (simulates realistic fall)
- **Terminal Velocity:** Cap at 600 pixels/s downward
- **Ground Detection:** Collision with platform layer

#### 4. Collectibles
- **Type:** Bones, treats, or stars
- **Behavior:** Static placement, respawn option
- **Collection:** Overlap detection (bounding box)
- **Feedback:** Simple particle effect + sound (future)

#### 5. Camera System
- **Type:** Side-scrolling follow camera
- **Follow Behavior:** Smooth lerp to player position
- **Boundaries:** Lock to level edges (no black space)

---

## Control System

### Desktop Controls (Keyboard)

| Key | Action |
|-----|--------|
| ← / A | Move Left |
| → / D | Move Right |
| ↑ / W / Space | Jump |
| P | Pause (future) |
| R | Restart Level (future) |

### Mobile Controls (Touch)

```
┌────────────────────────────────────┐
│                                    │
│         [Game View Area]           │
│                                    │
├──────────┬────────────┬────────────┤
│   ←      │     ↑      │     →      │
│  LEFT    │   JUMP     │   RIGHT    │
└──────────┴────────────┴────────────┘
```

- **Touch Zones:** Bottom 1/4 of screen divided into 3 zones
- **Alternative:** Virtual joystick (can be added)

### Input Manager Responsibilities
- Detect input type (keyboard vs touch)
- Normalize input to unified command system
- Handle simultaneous inputs (run + jump)
- Prevent conflicting commands

---

## Development Phases

### Phase 1: Foundation (Day 1-2)
**Goal:** Get basic Three.js setup with dog sprite visible

- [ ] Project file structure setup
- [ ] Three.js initialization (scene, camera, renderer)
- [ ] Load all 16 dog sprite frames
- [ ] Create sprite sheet or texture atlas
- [ ] Display single static dog sprite on screen
- [ ] Basic game loop running at 60 FPS

**Deliverable:** Dog appears on screen, game loop functional

---

### Phase 2: Animation System (Day 2-3)
**Goal:** Smooth sprite animation working

- [ ] Animation controller class
- [ ] Frame sequencer (timing, looping)
- [ ] Implement idle animation (frames 1-2)
- [ ] Implement run animation (frames 3-12)
- [ ] State-based animation switching
- [ ] Test animation transitions

**Deliverable:** Dog animates smoothly between idle and run states

---

### Phase 3: Player Movement (Day 3-4)
**Goal:** Keyboard controls for horizontal movement

- [ ] Input handler for keyboard events
- [ ] Player physics (velocity, acceleration)
- [ ] Horizontal movement implementation
- [ ] Animation triggers on movement
- [ ] Movement boundaries (screen edges)
- [ ] Face direction flipping (sprite mirror)

**Deliverable:** Dog runs left/right with keyboard, animations play correctly

---

### Phase 4: Jumping & Gravity (Day 4-5)
**Goal:** Vertical movement and physics

- [ ] Gravity system implementation
- [ ] Jump input detection
- [ ] Jump force application
- [ ] Parabolic jump arc
- [ ] Ground collision detection
- [ ] Landing logic

**Deliverable:** Dog can jump and land realistically

---

### Phase 5: Level & Platforms (Day 5-6)
**Goal:** Basic platformer level

- [ ] Platform class (position, size, collision box)
- [ ] Level data structure (JSON or hardcoded)
- [ ] Render platforms (simple rectangles or sprites)
- [ ] Collision detection (AABB)
- [ ] Collision response (stop falling, allow standing)
- [ ] Multi-platform level layout

**Deliverable:** Dog can stand on and jump between platforms

---

### Phase 6: Touch Controls (Day 6-7)
**Goal:** Mobile support

- [ ] Touch event listeners
- [ ] Virtual button zones
- [ ] Visual touch indicators (optional)
- [ ] Touch input normalization
- [ ] Test on mobile device or emulator
- [ ] Responsive canvas sizing

**Deliverable:** Game playable on mobile with touch controls

---

### Phase 7: Collectibles & Polish (Day 7-8)
**Goal:** Add game objectives and refinement

- [ ] Collectible item class
- [ ] Spawn collectibles in level
- [ ] Collision detection with collectibles
- [ ] Score counter UI
- [ ] Simple particle effect on collection
- [ ] Win condition (collect all items)
- [ ] Restart mechanism

**Deliverable:** Playable game loop with objective

---

### Phase 8: Camera & Scrolling (Day 8-9)
**Goal:** Proper level scrolling

- [ ] Camera follow logic
- [ ] Smooth camera lerp/easing
- [ ] Level boundaries enforcement
- [ ] Viewport management
- [ ] Parallax background (optional)

**Deliverable:** Camera follows player smoothly through level

---

## File Structure

```
dog_animate/
│
├── assets/
│   └── sprites/
│       ├── dog_1.png ... dog_16.png    # Existing frames
│       └── spritesheet.png             # (Optional: combined atlas)
│
├── src/
│   ├── main.js                         # Entry point, game initialization
│   ├── game/
│   │   ├── Game.js                     # Main game class
│   │   ├── GameLoop.js                 # Update/render loop
│   │   └── SceneManager.js             # Scene/level management
│   │
│   ├── player/
│   │   ├── Player.js                   # Player entity class
│   │   ├── AnimationController.js      # Sprite animation logic
│   │   └── PlayerPhysics.js            # Movement & physics
│   │
│   ├── level/
│   │   ├── Level.js                    # Level container
│   │   ├── Platform.js                 # Platform entity
│   │   └── Collectible.js              # Collectible items
│   │
│   ├── input/
│   │   ├── InputManager.js             # Unified input handler
│   │   ├── KeyboardController.js       # Keyboard events
│   │   └── TouchController.js          # Touch events
│   │
│   ├── camera/
│   │   └── Camera.js                   # Camera follow logic
│   │
│   └── utils/
│       ├── SpriteLoader.js             # Asset loading
│       ├── CollisionDetector.js        # AABB collision
│       └── Constants.js                # Game constants
│
├── index.html                          # HTML entry point
├── style.css                           # Minimal styling
└── GAME_STRATEGY_PLAN.md              # This document
```

---

## Technical Implementation Details

### Three.js 2D Setup

```javascript
// Orthographic camera for 2D view
const camera = new THREE.OrthographicCamera(
  width / -2, width / 2,
  height / 2, height / -2,
  1, 1000
);
camera.position.z = 500;

// Use sprites for 2D rendering
const sprite = new THREE.Sprite(material);
sprite.scale.set(spriteWidth, spriteHeight, 1);
```

### Animation Frame Management

```javascript
class AnimationController {
  constructor(frames, fps = 12) {
    this.frames = frames;          // Array of textures
    this.fps = fps;
    this.currentFrame = 0;
    this.frameTime = 1000 / fps;
    this.elapsed = 0;
  }

  update(deltaTime) {
    this.elapsed += deltaTime;
    if (this.elapsed >= this.frameTime) {
      this.currentFrame = (this.currentFrame + 1) % this.frames.length;
      this.elapsed = 0;
      return this.frames[this.currentFrame];
    }
    return null;
  }
}
```

### Physics System (Simple)

```javascript
class PlayerPhysics {
  constructor() {
    this.velocity = { x: 0, y: 0 };
    this.acceleration = 800;      // pixels/s²
    this.maxSpeed = 250;          // pixels/s
    this.gravity = 980;           // pixels/s²
    this.jumpForce = -500;        // pixels/s (negative = up)
    this.onGround = false;
  }

  update(deltaTime, inputX, inputJump) {
    // Horizontal movement
    this.velocity.x += inputX * this.acceleration * deltaTime;
    this.velocity.x = Math.max(-this.maxSpeed, Math.min(this.maxSpeed, this.velocity.x));

    // Apply friction when no input
    if (inputX === 0) {
      this.velocity.x *= 0.85;
    }

    // Gravity
    if (!this.onGround) {
      this.velocity.y += this.gravity * deltaTime;
    }

    // Jump
    if (inputJump && this.onGround) {
      this.velocity.y = this.jumpForce;
      this.onGround = false;
    }

    return this.velocity;
  }
}
```

### Collision Detection (AABB)

```javascript
function checkCollision(rect1, rect2) {
  return rect1.x < rect2.x + rect2.width &&
         rect1.x + rect1.width > rect2.x &&
         rect1.y < rect2.y + rect2.height &&
         rect1.y + rect1.height > rect2.y;
}
```

---

## Performance Considerations

### Optimization Strategies (MVP)
1. **Sprite Sheet:** Combine all 16 frames into single texture (reduces draw calls)
2. **Object Pooling:** Reuse collectible objects instead of creating/destroying
3. **Frustum Culling:** Only render objects in camera view
4. **Fixed Time Step:** Decouple physics from render rate
5. **Minimal DOM Updates:** Update score counter only on change

### Target Performance
- **60 FPS** on desktop (modern browsers)
- **30-60 FPS** on mobile (depends on device)
- **Load Time:** < 2 seconds for all assets

---

## Asset Requirements (Future)

### Graphics Needed
- Background layers (sky, clouds, hills) - parallax
- Platform tiles (grass, dirt, stone)
- Collectible sprites (bones, treats)
- UI elements (score, buttons)
- Particle effects (dust, sparkles)

### Audio Needed (Post-MVP)
- Jump sound effect
- Landing sound effect
- Collect item sound effect
- Background music (looping)
- Footstep sounds (optional)

---

## Testing Strategy

### Functionality Tests
- [ ] Dog loads and displays correctly
- [ ] All animations play smoothly
- [ ] Keyboard controls respond accurately
- [ ] Touch controls work on mobile
- [ ] Jump physics feel responsive
- [ ] Collision detection is reliable
- [ ] Collectibles register correctly
- [ ] Game runs at stable FPS

### Cross-Browser Testing
- Chrome (Desktop + Mobile)
- Firefox (Desktop)
- Safari (Desktop + iOS)
- Edge (Desktop)

### Device Testing
- Desktop (1920x1080, 1366x768)
- Tablet (768px width)
- Mobile (375px width)

---

## Known Limitations (MVP Scope)

### Not Included in MVP
- Sound effects / music
- Multiple levels
- Enemies or obstacles
- Animations for jump/land (using static frames)
- Power-ups or special abilities
- Menus (start, pause, settings)
- Leaderboard / save system
- Advanced particle effects

### Can Be Added Post-MVP
- All above features
- Level editor
- Multiplayer mode
- More dog animations (bark, sit command, etc.)
- Dynamic difficulty
- Achievement system

---

## Risk Assessment

| Risk | Probability | Impact | Mitigation |
|------|-------------|--------|------------|
| Animation framerate issues | Medium | Medium | Test different FPS values, allow configuration |
| Touch controls feel unresponsive | Medium | High | Increase button size, add visual feedback |
| Performance on low-end mobile | High | Medium | Implement quality settings, reduce particle count |
| Collision bugs (falling through platforms) | Medium | High | Careful physics timestep, add safety checks |
| Asset loading delays | Low | Low | Show loading screen, compress images |

---

## Success Metrics (MVP)

### Technical Metrics
- ✓ Game loads in < 3 seconds
- ✓ Maintains 60 FPS on desktop
- ✓ Maintains 30+ FPS on mobile
- ✓ No game-breaking bugs
- ✓ Works in 3+ browsers

### Gameplay Metrics
- ✓ Controls feel responsive (< 50ms input lag)
- ✓ Jump physics feel natural
- ✓ Player can complete level in < 2 minutes
- ✓ Collectibles are satisfying to gather

---

## Next Steps (Implementation Order)

### Immediate Actions
1. **Set up project structure** (create folders, files)
2. **Create index.html** with Three.js CDN or npm install
3. **Initialize Three.js** scene with orthographic camera
4. **Load first dog sprite** and render on screen
5. **Implement game loop** with delta time

### Week 1 Goals
- Complete Phases 1-3 (Foundation, Animation, Movement)
- Have a dog running on screen with keyboard controls
- Basic animation working

### Week 2 Goals
- Complete Phases 4-6 (Jumping, Platforms, Touch)
- Playable platformer prototype
- Mobile support functional

### Week 3 Goals (Polish)
- Complete Phases 7-8 (Collectibles, Camera)
- Refinement and testing
- Deploy demo version

---

## Questions for Client/Stakeholder

Before proceeding with implementation, please confirm:

1. **Visual Style:** Should we keep the exact look of the dog sprites, or do you want any visual effects applied (outlines, shadows, etc.)?

2. **Level Design:** Do you have a specific level layout in mind, or should we create a simple demo level?

3. **Collectibles:** What should the collectibles be? Bones, treats, stars, or something else?

4. **Difficulty:** Should the MVP be easy/casual or have some challenge (moving platforms, gaps)?

5. **Responsive Design:** Should the game fill the entire browser window or have a fixed aspect ratio?

6. **Deployment:** Where will this be hosted? (GitHub Pages, your own server, etc.)

---

## Resources & References

### Three.js 2D Resources
- [Three.js Docs - Sprite](https://threejs.org/docs/#api/en/objects/Sprite)
- [Three.js Docs - OrthographicCamera](https://threejs.org/docs/#api/en/cameras/OrthographicCamera)
- [Three.js Fundamentals](https://threejs.org/manual/#en/fundamentals)

### Game Development Patterns
- Game Loop pattern
- Component-based architecture
- State machines for animations
- AABB collision detection

### Sprite Animation
- Frame-by-frame sprite animation
- Texture atlases for performance
- Animation state machines

---

## Conclusion

This plan outlines a structured approach to building a dog platformer game using Three.js in 2D mode. The MVP focuses on core mechanics (running, jumping, collecting) with keyboard and touch controls, providing a solid foundation for future enhancements.

**Estimated Timeline:** 2-3 weeks for complete MVP
**Complexity:** Moderate (good for intermediate JS developers)
**Extensibility:** High (easy to add features incrementally)

Ready to proceed with implementation upon approval of this plan.

---

**Document Version:** 1.0
**Last Updated:** 2025-11-18
**Status:** Awaiting Approval
