# Vanilla HTML5 Canvas - Dog Platformer Game
## Simplified Implementation Plan

## Why Vanilla Canvas Instead of Three.js?

**Problems with Three.js approach:**
- ❌ Coordinate system confusion (Three.js uses center origin with Y-up)
- ❌ Overly complex for 2D sprite rendering
- ❌ Harder to debug position issues
- ❌ Large dependency (~600KB)

**Benefits of Vanilla Canvas:**
- ✅ Simple coordinate system (top-left origin, Y-down)
- ✅ Direct pixel control
- ✅ Easier debugging (can draw debug boxes easily)
- ✅ Zero dependencies
- ✅ Perfect for 2D sprite games

---

## Coordinate System

```
Canvas Coordinate System (Natural for Platformers):
(0, 0) ←─────────────────→ (1280, 0)
  ↓                           ↓
  │     SKY (top)             │
  │                           │
  │     PLATFORMS             │
  │                           │
  │     GROUND (bottom)       │
  ↓                           ↓
(0, 720) ←─────────────────→ (1280, 720)

Player Y position:
- SMALLER Y = HIGHER UP (jumping)
- LARGER Y = LOWER DOWN (falling)

Ground level = 620 (720 - 100 for ground height)
```

---

## Architecture - SIMPLE & PROVEN

### Single File Approach (game.js)
All game logic in ONE file for simplicity - no modules, no complexity

```
index.html
├── Simple HTML with canvas element
├── Inline CSS for styling
└── <script src="game.js"></script>  ← ALL GAME CODE HERE

game.js structure:
├── Global Constants
├── Asset Loader
├── Input Handler
├── Player Class
├── Platform Class
├── Collectible Class
├── Camera System
├── Game Loop
└── Initialization
```

---

## Core Implementation Strategy

### 1. Proven Coordinate Approach

```javascript
// GROUND LEVEL (where player should stand)
const GROUND_Y = 620;

// Player spawn position
player.x = 100;        // 100px from left
player.y = GROUND_Y - player.height;  // Standing ON ground

// Gravity (Y increases downward)
player.velocityY += GRAVITY;  // Falls DOWN (positive Y)
player.y += player.velocityY; // Move DOWN

// Platform collision (landing on top)
if (player.y + player.height >= platform.y &&
    player.bottom - velocityY <= platform.y) {
    // Player lands ON TOP of platform
    player.y = platform.y - player.height;
    player.velocityY = 0;
    player.onGround = true;
}
```

### 2. Simple Rendering

```javascript
// Clear canvas
ctx.clearRect(0, 0, canvas.width, canvas.height);

// Draw sky background
ctx.fillStyle = '#87CEEB';
ctx.fillRect(0, 0, canvas.width, canvas.height);

// Draw platforms (relative to camera)
for (platform of platforms) {
    ctx.fillStyle = '#8BC34A';
    ctx.fillRect(
        platform.x - camera.x,
        platform.y - camera.y,
        platform.width,
        platform.height
    );
}

// Draw player sprite (flipped if facing left)
if (player.facingLeft) {
    ctx.save();
    ctx.scale(-1, 1);
    ctx.drawImage(playerFrame, -(player.x - camera.x) - player.width, player.y - camera.y);
    ctx.restore();
} else {
    ctx.drawImage(playerFrame, player.x - camera.x, player.y - camera.y);
}
```

### 3. Foolproof Physics

```javascript
// Apply gravity EVERY FRAME
player.velocityY += GRAVITY * deltaTime;

// Cap fall speed
if (player.velocityY > MAX_FALL_SPEED) {
    player.velocityY = MAX_FALL_SPEED;
}

// Update position
player.x += player.velocityX * deltaTime;
player.y += player.velocityY * deltaTime;

// Check ground collision
if (player.y + player.height >= GROUND_Y) {
    player.y = GROUND_Y - player.height;
    player.velocityY = 0;
    player.onGround = true;
}
```

---

## File Structure (MINIMAL)

```
dog_animate/
├── index.html          (Canvas + basic HTML)
├── game.js             (ALL game code - single file)
├── dog_1.png ... dog_16.png  (sprites)
└── README.md
```

---

## Implementation Checklist

### Phase 1: Basic Setup (5 min)
- [ ] Create index.html with canvas
- [ ] Set canvas to 1280x720, maintain aspect ratio
- [ ] Basic styling (centered, responsive)
- [ ] Start game.js file

### Phase 2: Asset Loading (5 min)
- [ ] Load all 16 dog sprites
- [ ] Show loading progress bar
- [ ] Wait for all images to load before starting

### Phase 3: Player Rendering (5 min)
- [ ] Draw player sprite at fixed position
- [ ] Test sprite flipping (left/right)
- [ ] Verify sprite visible on canvas

### Phase 4: Input System (5 min)
- [ ] Keyboard event listeners (arrow keys, WASD, space)
- [ ] Touch button handlers
- [ ] Input state object

### Phase 5: Physics & Movement (10 min)
- [ ] Horizontal movement (left/right)
- [ ] Gravity implementation
- [ ] Jumping mechanics
- [ ] Ground collision
- [ ] Platform collision (AABB)

### Phase 6: Animation (5 min)
- [ ] Frame-by-frame animation
- [ ] State-based animation (idle, run, jump)
- [ ] Animation timing

### Phase 7: Camera (5 min)
- [ ] Follow player X position
- [ ] Keep player centered
- [ ] Clamp to level bounds

### Phase 8: Level & Collectibles (5 min)
- [ ] Platform array
- [ ] Collectible array
- [ ] Collision detection
- [ ] Score tracking

### Phase 9: UI & Polish (5 min)
- [ ] Score display
- [ ] Touch controls visibility
- [ ] Win condition

**Total Time: ~50 minutes for complete working game**

---

## Key Differences from Three.js Version

| Aspect | Three.js | Vanilla Canvas |
|--------|----------|----------------|
| Coordinate Origin | Center (0,0) | Top-Left (0,0) |
| Y-Axis | Up is negative | Down is positive |
| Rendering | WebGL sprites | drawImage() |
| Complexity | 15 files, modules | 1 file, no modules |
| Debugging | Hard to visualize | Easy console.log + draw debug |
| File Size | ~650KB total | ~50KB total |
| Setup Time | Complex | Immediate |

---

## Debug Strategy

```javascript
// Easy visual debugging in Canvas
function drawDebug() {
    // Draw player hitbox
    ctx.strokeStyle = 'red';
    ctx.strokeRect(player.x - camera.x, player.y - camera.y,
                   player.width, player.height);

    // Draw position text
    ctx.fillStyle = 'black';
    ctx.font = '16px Arial';
    ctx.fillText(`X: ${Math.floor(player.x)}, Y: ${Math.floor(player.y)}`, 10, 20);
    ctx.fillText(`VelY: ${Math.floor(player.velocityY)}`, 10, 40);
    ctx.fillText(`OnGround: ${player.onGround}`, 10, 60);
}
```

---

## Testing Plan

1. **Static Test**: Draw dog at (100, 500) - should be visible
2. **Gravity Test**: Dog should fall to ground and stop
3. **Jump Test**: Space bar should make dog jump up
4. **Movement Test**: Arrow keys should move dog left/right
5. **Platform Test**: Dog should land on elevated platforms
6. **Camera Test**: Camera should follow dog smoothly

---

## Success Criteria

- ✅ Dog visible on screen immediately
- ✅ Dog falls and lands on ground correctly
- ✅ Dog can jump and land
- ✅ Dog can walk left and right
- ✅ Dog can stand on platforms
- ✅ Camera follows dog smoothly
- ✅ Collectibles work
- ✅ No "flying off screen" bug

---

## Implementation Notes

**Canvas Best Practices:**
1. Use `requestAnimationFrame` for game loop
2. Calculate deltaTime for smooth movement
3. Clear canvas every frame
4. Draw from back to front (background → platforms → player → UI)
5. Use integer positions for crisp rendering

**Collision Detection:**
- AABB (Axis-Aligned Bounding Box)
- Check platform collision from top only
- Separate horizontal and vertical collision

**Performance:**
- No object creation in game loop
- Reuse objects
- Only draw visible elements

---

Ready to implement! This approach is proven, simple, and will definitely work.
