# Next Steps - Development Roadmap

## 🎯 Phase 2: Camera & Basic Player (NEXT - 2-3 Days)

### Goal
Add player sprite and implement camera system that follows the player.

### Tasks

#### 1. Create Camera Module (`src/camera.lua`)
```lua
-- Camera system with smooth following
-- Clamped to map boundaries
-- Support for zoom (optional)
```

**Implementation:**
- Camera position (x, y)
- Follow player with optional smoothing (lerp)
- Clamp to map boundaries (0 to mapWidth-screenWidth)
- Transform functions (apply/remove)

#### 2. Load Player Sprite
**Files:** Use `assets/sprites/dog/dog_5.png` as initial static sprite

**In main.lua:**
- Load dog sprite in `love.load()`
- Create player object with position, size
- Draw player at fixed position (100, 400)

#### 3. Implement Camera Following
- Camera tracks player.x and player.y
- Camera updates each frame
- Apply camera transform before drawing map/player
- UI remains fixed (drawn outside camera)

#### 4. Test Camera Boundaries
- Manually move player position (arrow keys → change x/y)
- Camera should follow but not go outside map
- Map edges should align with screen edges

### Expected Result
✓ Player sprite visible on map
✓ Camera follows player position
✓ Camera clamped to map boundaries
✓ Smooth camera movement

### Files to Create
- `src/camera.lua` - Camera class/module
- Update `main.lua` - Integrate camera

### Estimated Time
- 2-3 hours for camera implementation
- 1 hour for player sprite integration
- 1 hour for testing and debugging

---

## 🎯 Phase 3: Physics & Movement (3-5 Days)

### Goal
Player can move left/right, jump, and fall with gravity.

### Physics Constants (from HTML game)
```lua
PLAYER = {
    width = 80,
    height = 80,
    speed = 300,           -- px/s
    jumpForce = -850,      -- px/s (negative = upward)
    gravity = 1800,        -- px/s²
    maxFallSpeed = 800     -- px/s
}
```

### Tasks

#### 1. Implement Gravity
- Add `velocityY` to player
- Apply gravity each frame: `velocityY = velocityY + (gravity * dt)`
- Update position: `y = y + (velocityY * dt)`
- Clamp velocityY to maxFallSpeed

#### 2. Ground Detection (Simple)
- Check if `player.y >= GROUND_Y - player.height`
- If on ground: `velocityY = 0`, `onGround = true`

#### 3. Horizontal Movement
- Read keyboard input (A/D or Left/Right)
- Set `velocityX` based on input (-speed or +speed)
- Update position: `x = x + (velocityX * dt)`
- Track facing direction (for sprite flipping)

#### 4. Jumping
- Check for Space key press
- Only allow jump if `onGround == true`
- Set `velocityY = jumpForce` (negative for upward)

### Expected Result
✓ Player falls with gravity
✓ Player moves left/right with arrow keys
✓ Player can jump (reaches ~200px height)
✓ Ground collision works (doesn't fall through floor)

### Files to Create/Modify
- `src/player.lua` - Player class with physics
- Update `main.lua` - Integrate player physics

---

## 🎯 Phase 4: Collision System with Bump (2-3 Days)

### Goal
Proper collision detection with Tiled map solid objects using Bump.lua.

### Tasks

#### 1. Initialize Bump World
```lua
-- In love.load()
world = bump.newWorld(32)  -- Cell size 32px
```

#### 2. Load Map with Bump Plugin
```lua
map = sti("assets/maps/level1.lua", {"bump"})
map:bump_init(world)
```

#### 3. Add Player to Collision World
```lua
world:add(player, player.x, player.y, player.width, player.height)
```

#### 4. Use Collision Resolution
```lua
-- In love.update()
local actualX, actualY, cols, len = world:move(
    player,
    player.x,
    player.y,
    function(item, other)
        if other.properties and other.properties.collidable then
            return "slide"
        end
    end
)

player.x = actualX
player.y = actualY

-- Check if on ground
player.onGround = false
for i=1, len do
    if cols[i].normal.y == -1 then  -- Collision from below
        player.onGround = true
        player.velocityY = 0
    end
end
```

### Expected Result
✓ Player collides with ground from Tiled map
✓ Player can't fall through platforms
✓ Proper ground detection from collision normals
✓ Smooth sliding along surfaces

### Files to Modify
- Update `main.lua` - Bump initialization
- Update `src/player.lua` - Collision handling

---

## 🎯 Phase 5: Animation System (2-4 Days)

### Goal
Animated player sprite with state-based animations.

### Animation States
- **idle**: Frame 5 (or 6)
- **run**: Frames 5-8 cycling
- **jump**: Frame 7
- **fall**: Frame 8

### Tasks

#### 1. Load All Dog Sprites
```lua
dogSprites = {}
for i = 1, 20 do
    dogSprites[i] = love.graphics.newImage("assets/sprites/dog/dog_" .. i .. ".png")
end
```

#### 2. Create Animation Class
```lua
-- src/animation.lua
-- Frame-based animation at 12 FPS
-- Methods: new(), update(dt), getCurrentFrame()
```

#### 3. Create Animation Sets
```lua
animations = {
    idle = Animation.new({5}, 12),
    run = Animation.new({5, 6, 7, 8}, 12),
    jump = Animation.new({7}, 12),
    fall = Animation.new({8}, 12)
}
```

#### 4. Animation State Machine
```lua
-- Determine state
if not player.onGround then
    if player.velocityY < 0 then
        state = 'jump'
    else
        state = 'fall'
    end
elseif math.abs(player.velocityX) > 0 then
    state = 'run'
else
    state = 'idle'
end

-- Update animation
animations[state]:update(dt)
```

#### 5. Draw with Sprite Flipping
```lua
local scaleX = player.facingRight and 1 or -1
local offsetX = player.facingRight and 0 or player.width

love.graphics.draw(
    dogSprites[currentFrame],
    player.x + offsetX,
    player.y,
    0,  -- rotation
    scaleX, 1,  -- scale
    0, 0  -- origin
)
```

### Expected Result
✓ Player has smooth idle animation
✓ Running animation plays when moving
✓ Jump/fall animations trigger correctly
✓ Sprite flips based on movement direction

### Files to Create/Modify
- `src/animation.lua` - Animation helper class
- Update `src/player.lua` - Animation integration

---

## 🎯 Phase 6: Stats & UI (2-3 Days)

### Goal
Implement stats system (happiness, energy, level, XP, lives, score) and display HUD.

### Stats Structure
```lua
stats = {
    happiness = 100,
    energy = 100,
    level = 1,
    xp = 0,
    lives = 5,
    score = 0,
    bonusScore = 0
}
```

### Tasks

#### 1. Create Stats Module (`src/stats.lua`)
- Stats data structure
- Speed modifier calculation
- XP and leveling system
- Stat restoration/depletion

#### 2. Create UI Module (`src/ui.lua`)
- Draw HUD elements
- Lives (hearts)
- Level indicator
- Score display
- Happiness/energy bars

#### 3. Apply Speed Modifiers
```lua
local speedMod = stats:getSpeedModifier()
local actualSpeed = PLAYER.speed * speedMod
```

### Expected Result
✓ HUD displays all stats
✓ Speed affected by happiness/energy
✓ Level up system works
✓ Clean UI overlay

---

## 🎯 Remaining Phases (3-4 Weeks Total)

### Phase 7: Collectibles (2-3 days)
- Bones, balls, frisbees, bowls
- Collision detection
- Collection effects
- Score/stat updates

### Phase 8: NPCs & Interactions (2-3 days)
- 6 NPC dogs
- Interaction system
- Proximity detection
- Dialogue/effects

### Phase 9: Achievements (1-2 days)
- Achievement definitions
- Trigger checking
- Popup notifications
- Rewards

### Phase 10: Advanced Features (5-7 days)
- Moving platforms
- Ladders & climbing
- Checkpoints
- Powerups

### Phase 11: Polish & Audio (3-5 days)
- Sound effects
- Background music
- Particle effects
- Menus (title, pause, game over)
- Parallax backgrounds

---

## 📋 Quick Reference: File Creation Order

### Week 1
1. ✅ `conf.lua` - DONE
2. ✅ `main.lua` (Phase 1) - DONE
3. `src/camera.lua` - Phase 2
4. Update `main.lua` (camera integration) - Phase 2
5. `src/player.lua` - Phase 3
6. Update `main.lua` (player integration) - Phase 3

### Week 2
7. Update `main.lua` (Bump integration) - Phase 4
8. `src/animation.lua` - Phase 5
9. Update `src/player.lua` (animations) - Phase 5
10. `src/stats.lua` - Phase 6
11. `src/ui.lua` - Phase 6

### Week 3
12. `src/collectible.lua` - Phase 7
13. `src/npc.lua` - Phase 8
14. `src/achievement.lua` - Phase 9
15. `src/platform.lua` - Phase 10
16. `src/ladder.lua` - Phase 10
17. `src/checkpoint.lua` - Phase 10

### Week 4
18. `src/audio.lua` - Phase 11
19. `src/particles.lua` - Phase 11
20. `src/menu.lua` - Phase 11

---

## 🚀 Getting Started with Phase 2

To begin Phase 2, create `src/camera.lua`:

```lua
-- src/camera.lua
local Camera = {}
Camera.__index = Camera

function Camera.new(x, y, mapWidth, mapHeight, screenWidth, screenHeight)
    local self = setmetatable({}, Camera)
    self.x = x or 0
    self.y = y or 0
    self.mapWidth = mapWidth
    self.mapHeight = mapHeight
    self.screenWidth = screenWidth
    self.screenHeight = screenHeight
    return self
end

function Camera:follow(targetX, targetY)
    -- Center camera on target
    self.x = targetX - self.screenWidth / 2
    self.y = targetY - self.screenHeight / 2

    -- Clamp to map boundaries
    self.x = math.max(0, math.min(self.x, self.mapWidth - self.screenWidth))
    self.y = math.max(0, math.min(self.y, self.mapHeight - self.screenHeight))
end

function Camera:attach()
    love.graphics.push()
    love.graphics.translate(-self.x, -self.y)
end

function Camera:detach()
    love.graphics.pop()
end

return Camera
```

Then update `main.lua` to use it!

---

**Ready to code Phase 2!** 🎮
