-- ========================================
-- DOG ADVENTURE PLATFORMER - LUA EDITION
-- Phase 5: Animation System
-- ========================================

-- Global variables
local sti       -- Simple Tiled Implementation library
local map       -- The loaded Tiled map
local Camera    -- Camera module
local camera    -- Camera instance
local Player    -- Player module
local player    -- Player instance
local world     -- Bump collision world

local playerSprites = {}  -- Table to hold all 20 dog sprite frames

-- Constants
local WINDOW_WIDTH = 1280
local WINDOW_HEIGHT = 720

-- Debug flag
local showDebug = true

-- Load function - Called once at startup
function love.load()
    -- Set up graphics
    love.graphics.setDefaultFilter("nearest", "nearest")  -- Pixel-perfect scaling
    love.graphics.setBackgroundColor(0.53, 0.81, 0.92)   -- Sky blue (135, 206, 235)

    -- Load STI library
    print("Loading STI library...")
    sti = require("libraries.sti")

    -- Load Camera module
    print("Loading Camera module...")
    Camera = require("src.camera")

    -- Load Player module
    print("Loading Player module...")
    Player = require("src.player")

    -- Load the Tiled map WITH Bump plugin
    print("Loading Tiled map with Bump collision...")
    map = sti("assets/maps/level1.lua", {"bump"})

    if map then
        print("✓ Map loaded successfully!")
        print("  Map size: " .. map.width .. "x" .. map.height .. " tiles")
        print("  Tile size: " .. map.tilewidth .. "x" .. map.tileheight .. " pixels")
        print("  Total size: " .. (map.width * map.tilewidth) .. "x" .. (map.height * map.tileheight) .. " pixels")
        print("  Layers: " .. #map.layers)
        print("  Tilesets: " .. #map.tilesets)
        for i, tileset in ipairs(map.tilesets) do
            local imageStatus = tileset.image and "Image loaded" or "NO IMAGE"
            print("    Tileset " .. i .. ": " .. tileset.name .. " - " .. imageStatus)
        end
    else
        print("✗ Failed to load map!")
    end

    -- Initialize Bump collision world
    print("Initializing Bump collision world...")
    local bump = require("libraries.bump")
    world = bump.newWorld(32)  -- Cell size 32px (matches tile size)

    -- Hide the collision layer (solid objects layer)
    if map.layers.solid then
        map.layers.solid.visible = false
    end

    -- Register collision objects from Tiled map
    print("Registering map collision objects...")
    map:bump_init(world)

    -- Count collision objects for debugging
    local items, len = world:getItems()
    print("✓ Bump world initialized!")
    print("  Collision objects registered: " .. len)
    print("  Cell size: 32x32 pixels")

    -- Load all 20 dog sprite frames
    print("Loading player sprite frames...")
    for i = 1, 20 do
        playerSprites[i] = love.graphics.newImage("assets/sprites/dog/dog_" .. i .. ".png")
    end
    print("✓ Loaded " .. #playerSprites .. " sprite frames!")

    -- Create player object with physics and animations
    player = Player.new(100, 100, playerSprites)  -- Start at (100, 100)

    -- Add player to Bump world
    world:add(player, player.x, player.y, player.width, player.height)

    print("✓ Player created and added to Bump world!")
    print("  Starting position: (" .. player.x .. ", " .. player.y .. ")")
    print("  Size: " .. player.width .. "x" .. player.height)

    -- Initialize camera
    local mapPixelWidth = map.width * map.tilewidth
    local mapPixelHeight = map.height * map.tileheight

    camera = Camera.new(
        0, 0,                   -- Initial position
        mapPixelWidth,          -- Map width (1216px)
        mapPixelHeight,         -- Map height (960px)
        WINDOW_WIDTH,           -- Screen width
        WINDOW_HEIGHT           -- Screen height
    )

    print("✓ Camera initialized!")

    -- Print welcome message
    print("\n========================================")
    print("DOG ADVENTURE PLATFORMER - LUA EDITION")
    print("Phase 5: Animation System")
    print("========================================")
    print("\nControls:")
    print("  Left/Right (A/D): Move horizontally")
    print("  Space: Jump")
    print("  F: Toggle debug visualization")
    print("  Esc: Quit")
    print("\nAnimated dog sprite with state-based animations!")
    print("Watch the dog run, jump, and fall!")
    print("========================================\n")
end

-- Update function - Called every frame
function love.update(dt)
    -- Update the map (for animated tiles, etc.)
    if map then
        map:update(dt)
    end

    -- Gather input for player
    local keys = {
        left = love.keyboard.isDown("left") or love.keyboard.isDown("a"),
        right = love.keyboard.isDown("right") or love.keyboard.isDown("d"),
        jump = love.keyboard.isDown("space")
    }

    -- Update player physics (calculates new position and velocity)
    player:updatePhysics(dt, keys)

    -- Keep player within map bounds (before collision check)
    local mapPixelWidth = map.width * map.tilewidth
    local mapPixelHeight = map.height * map.tileheight
    player:clampToMap(mapPixelWidth, mapPixelHeight)

    -- Handle collision with Bump
    local goalX = player.x
    local goalY = player.y

    -- Collision filter function
    local function playerFilter(item, other)
        if other.properties and other.properties.collidable then
            return "slide"  -- Slide along solid objects
        end
        return nil  -- Pass through non-collidable objects
    end

    -- Move player and resolve collisions
    local actualX, actualY, cols, len = world:move(
        player,
        goalX,
        goalY,
        playerFilter
    )

    -- Update player position to collision-resolved position
    player.x = actualX
    player.y = actualY

    -- Check ground collision (from collision normals)
    player.onGround = false

    for i = 1, len do
        local col = cols[i]

        -- Check if collision is from below (standing on platform)
        if col.normal.y == -1 then
            player.onGround = true
            player.velocityY = 0
        end

        -- Check if collision is from above (hitting ceiling)
        if col.normal.y == 1 then
            player.velocityY = 0
        end

        -- Optional: Check wall collisions
        if col.normal.x ~= 0 then
            player.velocityX = 0
        end
    end

    -- Update animation AFTER collision (so onGround is set correctly)
    player:updateAnimation(dt)

    -- Update camera to follow player
    local playerCenterX, playerCenterY = player:getCenter()
    camera:follow(playerCenterX, playerCenterY, dt)
end

-- Draw function - Called every frame
function love.draw()
    -- Apply camera transformation
    camera:attach()

    -- Draw the Tiled map
    if map then
        map:draw()
    end

    -- Draw player
    player:draw()

    -- Draw debug visualization if enabled
    if showDebug then
        player:drawDebug()

        -- Draw all collision boxes from Bump world
        love.graphics.setColor(1, 0, 0, 0.3)  -- Red with transparency
        local items, len = world:getItems()
        for i = 1, len do
            local item = items[i]
            if item ~= player then  -- Don't draw player collision box twice
                local x, y, w, h = world:getRect(item)
                love.graphics.rectangle("line", x, y, w, h)
            end
        end
        love.graphics.setColor(1, 1, 1, 1)  -- Reset color
    end

    -- Remove camera transformation
    camera:detach()

    -- Draw UI overlay (not affected by camera)
    love.graphics.setColor(0, 0, 0, 0.7)
    love.graphics.rectangle("fill", 0, 0, 450, 260)
    love.graphics.setColor(1, 1, 1, 1)

    love.graphics.print("DOG ADVENTURE PLATFORMER - LUA", 10, 10)
    love.graphics.print("Phase 5: Animation System", 10, 30)
    love.graphics.print("FPS: " .. love.timer.getFPS(), 10, 50)

    if map then
        love.graphics.print("Map: " .. map.width .. "x" .. map.height .. " tiles", 10, 70)
    end

    -- Player physics info
    love.graphics.print(
        string.format("Position: (%.0f, %.0f)", player.x, player.y),
        10,
        90
    )
    love.graphics.print(
        string.format("Velocity: (%.0f, %.0f)", player.velocityX, player.velocityY),
        10,
        110
    )
    love.graphics.print(
        "State: " .. player:getState(),
        10,
        130
    )
    love.graphics.print(
        "Animation Frame: " .. player:getCurrentFrame(),
        10,
        150
    )
    love.graphics.print(
        "On Ground: " .. (player.onGround and "YES" or "NO"),
        10,
        170
    )
    love.graphics.print(
        "Facing: " .. (player.facingRight and "RIGHT" or "LEFT"),
        10,
        190
    )

    -- Bump info
    local items, len = world:getItems()
    love.graphics.print(
        "Collision Objects: " .. len,
        10,
        210
    )
    love.graphics.print(
        "Debug: " .. (showDebug and "ON (F to toggle)" or "OFF (F to toggle)"),
        10,
        230
    )

    -- Reset color
    love.graphics.setColor(1, 1, 1, 1)
end

-- Keyboard input
function love.keypressed(key)
    if key == "escape" then
        love.event.quit()
    end

    if key == "f" then
        showDebug = not showDebug
        print("Debug visualization: " .. (showDebug and "ON" or "OFF"))
    end
end

-- Handle window resize
function love.resize(w, h)
    if map then
        map:resize(w, h)
    end

    -- Update camera screen dimensions
    if camera then
        camera.screenWidth = w
        camera.screenHeight = h
    end
end
