-- ========================================
-- ANIMATION MODULE
-- Handles frame-based sprite animations
-- ========================================

local Animation = {}
Animation.__index = Animation

-- Create a new animation
-- @param frames: Array of frame indices (e.g., {1, 2, 3, 4})
-- @param fps: Frames per second (default: 12)
-- @param loop: Should animation loop? (default: true)
function Animation.new(frames, fps, loop)
    local self = setmetatable({}, Animation)

    self.frames = frames or {1}
    self.fps = fps or 12
    self.loop = loop == nil and true or loop

    self.currentFrame = 1
    self.timer = 0
    self.frameDuration = 1 / self.fps
    self.isPlaying = true
    self.isFinished = false

    return self
end

-- Update animation timer
-- @param dt: Delta time
function Animation:update(dt)
    if not self.isPlaying or self.isFinished then
        return
    end

    self.timer = self.timer + dt

    -- Advance to next frame when timer exceeds frame duration
    if self.timer >= self.frameDuration then
        self.timer = self.timer - self.frameDuration
        self.currentFrame = self.currentFrame + 1

        -- Handle loop or finish
        if self.currentFrame > #self.frames then
            if self.loop then
                self.currentFrame = 1
            else
                self.currentFrame = #self.frames
                self.isFinished = true
                self.isPlaying = false
            end
        end
    end
end

-- Get the current frame index from the frames array
-- @return: Frame index (e.g., if frames={5,6,7,8} and currentFrame=2, returns 6)
function Animation:getCurrentFrame()
    return self.frames[self.currentFrame]
end

-- Reset animation to first frame
function Animation:reset()
    self.currentFrame = 1
    self.timer = 0
    self.isFinished = false
    self.isPlaying = true
end

-- Pause animation
function Animation:pause()
    self.isPlaying = false
end

-- Resume animation
function Animation:resume()
    if not self.isFinished then
        self.isPlaying = true
    end
end

-- Check if animation is finished (only relevant for non-looping animations)
function Animation:hasFinished()
    return self.isFinished
end

-- Get total animation duration in seconds
function Animation:getDuration()
    return #self.frames * self.frameDuration
end

-- Clone animation (creates a new instance with same parameters)
function Animation:clone()
    return Animation.new(self.frames, self.fps, self.loop)
end

return Animation
