-- ========================================
-- CAMERA MODULE
-- Smooth camera following with map boundary clamping
-- ========================================

local Camera = {}
Camera.__index = Camera

-- Create a new camera
-- @param x: Initial camera x position
-- @param y: Initial camera y position
-- @param mapWidth: Total map width in pixels
-- @param mapHeight: Total map height in pixels
-- @param screenWidth: Viewport width (window width)
-- @param screenHeight: Viewport height (window height)
function Camera.new(x, y, mapWidth, mapHeight, screenWidth, screenHeight)
    local self = setmetatable({}, Camera)

    self.x = x or 0
    self.y = y or 0
    self.mapWidth = mapWidth
    self.mapHeight = mapHeight
    self.screenWidth = screenWidth
    self.screenHeight = screenHeight

    -- Smoothing factor (0 = instant, 1 = no movement)
    -- Lower = faster following, Higher = smoother/slower
    self.smoothness = 0.1

    -- Target position (what the camera wants to follow)
    self.targetX = self.x
    self.targetY = self.y

    return self
end

-- Update camera to follow a target position
-- @param targetX: X position to follow (usually player.x)
-- @param targetY: Y position to follow (usually player.y)
-- @param dt: Delta time (optional, for smooth lerp)
function Camera:follow(targetX, targetY, dt)
    -- Center camera on target
    self.targetX = targetX - self.screenWidth / 2
    self.targetY = targetY - self.screenHeight / 2

    -- Clamp target to map boundaries
    self.targetX = math.max(0, math.min(self.targetX, self.mapWidth - self.screenWidth))
    self.targetY = math.max(0, math.min(self.targetY, self.mapHeight - self.screenHeight))

    if dt then
        -- Smooth lerp towards target
        self.x = self.x + (self.targetX - self.x) * (1 - self.smoothness)
        self.y = self.y + (self.targetY - self.y) * (1 - self.smoothness)
    else
        -- Instant snap to target
        self.x = self.targetX
        self.y = self.targetY
    end
end

-- Set camera position directly (no smoothing)
function Camera:setPosition(x, y)
    self.x = x
    self.y = y
    self.targetX = x
    self.targetY = y

    -- Clamp to boundaries
    self.x = math.max(0, math.min(self.x, self.mapWidth - self.screenWidth))
    self.y = math.max(0, math.min(self.y, self.mapHeight - self.screenHeight))
end

-- Apply camera transformation (call before drawing world)
function Camera:attach()
    love.graphics.push()
    love.graphics.translate(-self.x, -self.y)
end

-- Remove camera transformation (call after drawing world)
function Camera:detach()
    love.graphics.pop()
end

-- Get camera bounds (for culling checks)
function Camera:getBounds()
    return self.x, self.y, self.screenWidth, self.screenHeight
end

-- Check if a rectangle is visible in camera view
-- Useful for culling objects outside camera
function Camera:isVisible(x, y, width, height)
    return x + width > self.x and
           x < self.x + self.screenWidth and
           y + height > self.y and
           y < self.y + self.screenHeight
end

-- Convert screen coordinates to world coordinates
function Camera:toWorld(screenX, screenY)
    return screenX + self.x, screenY + self.y
end

-- Convert world coordinates to screen coordinates
function Camera:toScreen(worldX, worldY)
    return worldX - self.x, worldY - self.y
end

return Camera
