-- ========================================
-- PLAYER MODULE
-- Player character with physics, movement, and state management
-- ========================================

local Animation = require("src.animation")

local Player = {}
Player.__index = Player

-- Physics constants (from HTML game analysis)
local PLAYER_WIDTH = 80
local PLAYER_HEIGHT = 80
local PLAYER_SPEED = 300        -- Base horizontal speed (px/s)
local JUMP_FORCE = -850         -- Initial jump velocity (negative = upward)
local GRAVITY = 1800            -- Gravity acceleration (px/s²)
local MAX_FALL_SPEED = 800      -- Terminal velocity (px/s)
local GROUND_Y = 620            -- Ground level (from top of screen)

-- Create a new player
-- @param x: Starting X position
-- @param y: Starting Y position
-- @param sprites: Array of 20 player sprite images
function Player.new(x, y, sprites)
    local self = setmetatable({}, Player)

    -- Position
    self.x = x
    self.y = y
    self.width = PLAYER_WIDTH
    self.height = PLAYER_HEIGHT

    -- Velocity
    self.velocityX = 0
    self.velocityY = 0

    -- State
    self.onGround = false
    self.facingRight = true

    -- Graphics
    self.sprites = sprites

    -- Animation system
    -- Frame indices based on dog sprite sheet
    self.animations = {
        idle = Animation.new({5}, 12, true),             -- Frame 5 only (static idle)
        run = Animation.new({5, 6, 7, 4, 3}, 12, true),  -- Frames 5,6,7,4,3 at 12 FPS (run cycle)
        jump = Animation.new({7}, 12, false),             -- Frame 7 (single frame, no loop)
        fall = Animation.new({3}, 12, false)              -- Frame 3 (single frame, no loop)
    }

    self.currentAnimation = self.animations.idle
    self.previousState = 'idle'

    -- Input state
    self.jumpPressed = false
    self.jumpReleased = true  -- For jump buffering

    return self
end

-- Update player physics and movement (Phase 4: Bump collision handled in main.lua)
-- @param dt: Delta time
-- @param keys: Table of key states {left, right, jump}
function Player:updatePhysics(dt, keys)
    -- Horizontal movement
    self.velocityX = 0

    if keys.left and not keys.right then
        self.velocityX = -PLAYER_SPEED
        self.facingRight = false
    elseif keys.right and not keys.left then
        self.velocityX = PLAYER_SPEED
        self.facingRight = true
    end

    -- Apply gravity
    if not self.onGround then
        self.velocityY = self.velocityY + (GRAVITY * dt)

        -- Clamp to max fall speed
        if self.velocityY > MAX_FALL_SPEED then
            self.velocityY = MAX_FALL_SPEED
        end
    end

    -- Jumping
    if keys.jump and self.onGround and self.jumpReleased then
        self.velocityY = JUMP_FORCE
        self.onGround = false
        self.jumpPressed = true
        self.jumpReleased = false
    end

    -- Track jump release for next jump
    if not keys.jump then
        self.jumpReleased = true
        self.jumpPressed = false
    end

    -- Update position (Bump collision resolution happens in main.lua)
    self.x = self.x + (self.velocityX * dt)
    self.y = self.y + (self.velocityY * dt)

    -- Note: onGround state is set in main.lua after Bump collision check
    -- Note: Animation is updated in updateAnimation() after collision
end

-- Update animation based on current state
-- Called AFTER collision resolution so onGround is correct
function Player:updateAnimation(dt)
    local currentState = self:getState()

    -- Switch animation if state changed
    if currentState ~= self.previousState then
        self.currentAnimation = self.animations[currentState]
        self.currentAnimation:reset()
        self.previousState = currentState
    end

    -- Update current animation
    self.currentAnimation:update(dt)
end

-- Keep player within map boundaries
-- @param mapWidth: Map width in pixels
-- @param mapHeight: Map height in pixels
function Player:clampToMap(mapWidth, mapHeight)
    self.x = math.max(0, math.min(self.x, mapWidth - self.width))
    self.y = math.max(0, math.min(self.y, mapHeight - self.height))
end

-- Draw the player sprite with current animation frame
function Player:draw()
    local scaleX = self.facingRight and 1 or -1
    local offsetX = self.facingRight and 0 or self.width

    -- Get current animation frame
    local frameIndex = self.currentAnimation:getCurrentFrame()
    local sprite = self.sprites[frameIndex]

    love.graphics.draw(
        sprite,
        self.x + offsetX,
        self.y,
        0,  -- rotation
        scaleX * (self.width / sprite:getWidth()),   -- scaleX (with flip)
        self.height / sprite:getHeight(),            -- scaleY
        0,  -- originX
        0   -- originY
    )
end

-- Draw debug visualization
function Player:drawDebug()
    -- Bounding box
    love.graphics.setColor(0, 1, 0, 0.3)  -- Green with transparency
    love.graphics.rectangle("line", self.x, self.y, self.width, self.height)

    -- Center point
    love.graphics.setColor(1, 0, 0, 1)  -- Red
    love.graphics.circle("fill", self.x + self.width / 2, self.y + self.height / 2, 3)

    -- Ground indicator
    if self.onGround then
        love.graphics.setColor(0, 1, 0, 1)  -- Green
        love.graphics.print("ON GROUND", self.x, self.y - 20)
    else
        love.graphics.setColor(1, 0, 0, 1)  -- Red
        love.graphics.print("IN AIR", self.x, self.y - 20)
    end

    -- Velocity vector
    love.graphics.setColor(1, 1, 0, 1)  -- Yellow
    local centerX = self.x + self.width / 2
    local centerY = self.y + self.height / 2
    love.graphics.line(
        centerX,
        centerY,
        centerX + self.velocityX * 0.1,  -- Scale down for visibility
        centerY + self.velocityY * 0.1
    )

    -- Reset color
    love.graphics.setColor(1, 1, 1, 1)
end

-- Get player center position (for camera following)
function Player:getCenter()
    return self.x + self.width / 2, self.y + self.height / 2
end

-- Get current state for animation system (Phase 5)
function Player:getState()
    if not self.onGround then
        if self.velocityY < 0 then
            return 'jump'
        else
            return 'fall'
        end
    elseif math.abs(self.velocityX) > 0 then
        return 'run'
    else
        return 'idle'
    end
end

-- Get current animation frame number (for debug display)
function Player:getCurrentFrame()
    return self.currentAnimation:getCurrentFrame()
end

return Player
