// ========================================
// DOG ADVENTURE PLATFORMER - VANILLA CANVAS
// All game code in one file for simplicity
// ========================================
//
// SPRITE SHEET LAYOUT (4x4 grid, 16 frames total):
// dog_1 to dog_16.png organized as:
//   Row 1 (frames 1-4):   Front sitting view (not used in platformer)
//   Row 2 (frames 5-8):   Side-view running cycle (MAIN ANIMATIONS)
//   Row 3 (frames 9-12):  Back/turning views
//   Row 4 (frames 13-16): Back/turning views
//
// For platformer gameplay, we use Row 2 (side view) for all animations
//
// ========================================

// ===== CONSTANTS =====
const CANVAS_WIDTH = 1280;
const CANVAS_HEIGHT = 720;
const GROUND_Y = 620; // Ground level (620px from top)

const PLAYER = {
    width: 80,
    height: 80,
    speed: 300,
    jumpForce: -850,    // Increased from -600 to allow ~200px jump height
    gravity: 1800,
    maxFallSpeed: 800
};

const ANIMATION_FPS = 12;
const FRAME_TIME = 1000 / ANIMATION_FPS;

// ===== GLOBAL VARIABLES =====
const canvas = document.getElementById('gameCanvas');
const ctx = canvas.getContext('2d');

let dogSprites = [];
let boneSprites = [];
let cloudSprites = [];
let sunSprite = null;
let treeSprites = [];
let ballSprite = null;
let frisbeeSprite = null;
let hydrantSprite = null;
let bedSprite = null;
let npcDogSprites = []; // NPC dogs (dog_1_ through dog_6_ from props)
let happyStatSprite = null; // Phase 4 UI
let energyStatSprite = null;
let itemsCollectedSprite = null;
let waterBowlSprite = null;
let foodBowlSprite = null;
let loadedSprites = 0;
let totalSprites = 20;
let totalBones = 3;
let totalClouds = 3;
let totalTrees = 2;
let totalNPCDogs = 6;

let gameRunning = false;
let lastTime = 0;
let score = 0;
let useFrameSet2 = false; // Toggle between frame sets (1-16 vs 17-20)

// Phase 4: Stats System
const stats = {
    happiness: 100,
    maxHappiness: 100,
    energy: 100,
    maxEnergy: 100,
    level: 1,
    xp: 0,
    idleTime: 0,
    lives: 5,        // New: Lives system
    maxLives: 5,
    bonusScore: 0    // New: Bonus score separate from main score
};

const collectionStats = {
    bonesCollected: 0,
    bonesTotal: 12,
    ballsCollected: 0,
    ballsTotal: 4,
    frisbeesCollected: 0,
    frisbeesTotal: 2,
    npcsGreeted: 0,
    npcsTotal: 6,
    squirrelCaught: false,
    waterBowlsCollected: 0,
    foodBowlsCollected: 0
};

const achievements = [];
const achievementDefs = [
    { id: 'first_friend', name: 'First Friend', desc: 'Meet 1 NPC', icon: '🤝', unlocked: false },
    { id: 'social_butterfly', name: 'Social Butterfly', desc: 'Meet all 6 NPCs', icon: '🦋', unlocked: false },
    { id: 'bone_collector', name: 'Bone Collector', desc: 'Collect all 12 bones', icon: '🦴', unlocked: false },
    { id: 'toy_master', name: 'Toy Master', desc: 'Collect all balls & frisbees', icon: '🎾', unlocked: false },
    { id: 'squirrel_whisperer', name: 'Squirrel Whisperer', desc: 'Catch the squirrel', icon: '🐿️', unlocked: false },
    { id: 'perfect_score', name: 'Perfect Score', desc: 'Reach 400 points', icon: '🏆', unlocked: false }
];

let showCollectionTracker = false;
let achievementPopup = null; // { achievement, timeLeft }

// Powerup system
const powerup = {
    active: false,
    type: null, // 'speed', 'invincible', etc.
    timeLeft: 0,
    duration: 8, // seconds
    speedMultiplier: 2.0 // 2x speed boost
};

// Input state
const keys = {
    left: false,
    right: false,
    up: false, // Phase 5: For ladder climbing
    down: false, // Phase 5: For ladder climbing
    jump: false,
    jumpPressed: false
};

// Player object
const player = {
    x: 100,
    y: 400, // Will be set properly after platforms are created
    width: PLAYER.width,
    height: PLAYER.height,
    velocityX: 0,
    velocityY: 0,
    onGround: false,
    facingRight: true,
    currentFrame: 0,
    animationTime: 0,
    animation: 'idle', // idle, run, jump, fall, climbing
    currentPlatform: null, // Track which platform player is standing on
    climbing: false, // Is player on a ladder?
    currentLadder: null // Which ladder is player on?
};

// Camera
const camera = {
    x: 0,
    y: 0
};

// Game objects
const platforms = [];
const movingPlatforms = []; // Phase 5: Moving platforms (horizontal, vertical, circular)
const ladders = []; // Phase 5: Climbable ladders
const slopes = []; // Phase 5: Angled ramps
const collectibles = [];
const clouds = [];
const trees = [];
const obstacles = [];
const checkpoints = [];
let currentCheckpoint = 0; // Index of last activated checkpoint
const npcs = []; // NPC dogs
const particles = []; // Visual effects (hearts, etc)
let squirrel = null; // Squirrel chase mini-game
const backgroundLayers = []; // Phase 5: Multi-layer parallax backgrounds
const foregroundLayers = []; // Phase 5: Foreground parallax elements
let flowerSprites = []; // Phase 5: Foreground flower sprites
const zones = []; // Phase 5: Themed zones with visual transitions
let currentZone = 0; // Current zone player is in
const interactiveElements = []; // Phase 5: Switches, doors, bridges
const waterZones = []; // Phase 5: Swimming areas

// ===== ASSET LOADING =====
function loadAssets() {
    const loadingProgress = document.getElementById('loadingProgress');
    const totalFlowers = 3; // Phase 5: 3 flower variants
    const totalAssets = totalSprites + totalBones + totalClouds + 1 + totalTrees + 4 + totalNPCDogs + 5 + totalFlowers; // +3 Phase 5 flowers
    let assetsLoaded = 0;

    // Base path for assets - essential for SvelteKit routing
    const basePath = '/games/dog_animate/';

    const updateProgress = () => {
        assetsLoaded++;
        const progress = (assetsLoaded / totalAssets) * 100;
        loadingProgress.style.width = `${progress}%`;

        if (assetsLoaded === totalAssets) {
            document.getElementById('loading').classList.add('hidden');
            initGame();
        }
    };

    // Load dog sprites
    for (let i = 1; i <= totalSprites; i++) {
        const img = new Image();
        img.src = `${basePath}dog_${i}.png`;
        img.onload = updateProgress;
        img.onerror = () => {
            console.error(`Failed to load dog_${i}.png`);
            updateProgress(); // Proceed anyway
        };
        dogSprites.push(img);
    }

    // Load bone sprites from props folder
    for (let i = 1; i <= totalBones; i++) {
        const img = new Image();
        img.src = `${basePath}props/bone_${i}_.png`;
        img.onload = updateProgress;
        img.onerror = () => {
            console.error(`Failed to load props/bone_${i}_.png`);
            updateProgress();
        };
        boneSprites.push(img);
    }

    // Load cloud sprites
    for (let i = 1; i <= totalClouds; i++) {
        const img = new Image();
        img.src = `${basePath}props/cloud_${i}_.png`;
        img.onload = updateProgress;
        img.onerror = () => {
            console.error(`Failed to load props/cloud_${i}_.png`);
            updateProgress();
        };
        cloudSprites.push(img);
    }

    // Load sun sprite
    sunSprite = new Image();
    sunSprite.src = `${basePath}props/sun_.png`;
    sunSprite.onload = updateProgress;
    sunSprite.onerror = () => {
        console.error('Failed to load props/sun_.png');
        updateProgress();
    };

    // Load tree sprites
    for (let i = 1; i <= totalTrees; i++) {
        const img = new Image();
        img.src = `${basePath}props/tree_${i}_.png`;
        img.onload = updateProgress;
        img.onerror = () => {
            console.error(`Failed to load props/tree_${i}_.png`);
            updateProgress();
        };
        treeSprites.push(img);
    }

    // Load interactive prop sprites (Phase 2)
    ballSprite = new Image();
    ballSprite.src = `${basePath}props/ball_1_.png`;
    ballSprite.onload = updateProgress;
    ballSprite.onerror = () => {
        console.error('Failed to load props/ball_1_.png');
        updateProgress();
    };

    frisbeeSprite = new Image();
    frisbeeSprite.src = `${basePath}props/frisbee_1_.png`;
    frisbeeSprite.onload = updateProgress;
    frisbeeSprite.onerror = () => {
        console.error('Failed to load props/frisbee_1_.png');
        updateProgress();
    };

    hydrantSprite = new Image();
    hydrantSprite.src = `${basePath}props/fire_hydrant_1_.png`;
    hydrantSprite.onload = updateProgress;
    hydrantSprite.onerror = () => {
        console.error('Failed to load props/fire_hydrant_1_.png');
        updateProgress();
    };

    bedSprite = new Image();
    bedSprite.src = `${basePath}props/bed_1_.png`;
    bedSprite.onload = updateProgress;
    bedSprite.onerror = () => {
        console.error('Failed to load props/bed_1_.png');
        updateProgress();
    };

    // Load NPC dog sprites (Phase 3)
    for (let i = 1; i <= totalNPCDogs; i++) {
        const img = new Image();
        img.src = `${basePath}props/dog_${i}_.png`;
        img.onload = updateProgress;
        img.onerror = () => {
            console.error(`Failed to load props/dog_${i}_.png`);
            updateProgress();
        };
        npcDogSprites.push(img);
    }

    // Load Phase 4 UI sprites
    happyStatSprite = new Image();
    happyStatSprite.src = `${basePath}props/happy_stat_.png`;
    happyStatSprite.onload = updateProgress;
    happyStatSprite.onerror = () => {
        console.error('Failed to load props/happy_stat_.png');
        updateProgress();
    };

    energyStatSprite = new Image();
    energyStatSprite.src = `${basePath}props/enegy_stat_.png`;
    energyStatSprite.onload = updateProgress;
    energyStatSprite.onerror = () => {
        console.error('Failed to load props/enegy_stat_.png');
        updateProgress();
    };

    itemsCollectedSprite = new Image();
    itemsCollectedSprite.src = `${basePath}props/items_collected.png`;
    itemsCollectedSprite.onload = updateProgress;
    itemsCollectedSprite.onerror = () => {
        console.error('Failed to load props/items_collected.png');
        updateProgress();
    };

    waterBowlSprite = new Image();
    waterBowlSprite.src = `${basePath}props/water_bowel_1_.png`;
    waterBowlSprite.onload = updateProgress;
    waterBowlSprite.onerror = () => {
        console.error('Failed to load props/water_bowel_1_.png');
        updateProgress();
    };

    foodBowlSprite = new Image();
    foodBowlSprite.src = `${basePath}props/food_bowel_1_.png`;
    foodBowlSprite.onload = updateProgress;
    foodBowlSprite.onerror = () => {
        console.error('Failed to load props/food_bowel_1_.png');
        updateProgress();
    };

    // Load flower sprites (Phase 5: foreground parallax)
    for (let i = 1; i <= 3; i++) {
        const img = new Image();
        img.src = `${basePath}props/flower_${i}_.png`;
        img.onload = updateProgress;
        img.onerror = () => {
            console.error(`Failed to load props/flower_${i}_.png`);
            updateProgress();
        };
        flowerSprites.push(img);
    }
}

// ===== GAME INITIALIZATION =====
function initGame() {
    console.log('🐕 Dog Adventure Platformer - Phase 5 Loaded!');
    console.log('');
    console.log('=== ASSETS ===');
    console.log(`🦴 Bone collectibles: ${boneSprites.length} variants`);
    console.log(`⚾ Ball: ${ballSprite ? 'Loaded' : 'Missing'} | 🥏 Frisbee: ${frisbeeSprite ? 'Loaded' : 'Missing'}`);
    console.log(`💧 Water Bowl: ${waterBowlSprite ? 'Loaded' : 'Missing'} | 🍖 Food Bowl: ${foodBowlSprite ? 'Loaded' : 'Missing'}`);
    console.log(`🚒 Fire Hydrant: ${hydrantSprite ? 'Loaded' : 'Missing'} | 🛏️ Dog Bed: ${bedSprite ? 'Loaded' : 'Missing'}`);
    console.log(`🐶 NPC Dogs: ${npcDogSprites.length} loaded, ${npcs.length} placed`);
    console.log(`🐿️ Squirrel: ${squirrel ? 'Ready to chase!' : 'Missing'}`);
    console.log(`☁️ Clouds: ${cloudSprites.length} variants, ${clouds.length} instances`);
    console.log(`🌳 Trees: ${treeSprites.length} variants, ${trees.length} instances`);
    console.log(`☀️ Sun: ${sunSprite ? 'Loaded' : 'Missing'}`);
    console.log('');
    console.log('=== COLLECTIBLES & SCORING ===');
    console.log(`💯 Total collectibles: ${collectibles.length} | Max Score: 400 points`);
    console.log(`  ├─ Bones (10pts each): 12`);
    console.log(`  ├─ Balls (20pts each): 4`);
    console.log(`  ├─ Frisbees (30pts each): 2`);
    console.log(`  ├─ Water Bowls (+30 energy): 3`);
    console.log(`  ├─ Food Bowls (+50 energy): 2`);
    console.log(`  ├─ NPCs (15pts each): 6`);
    console.log(`  └─ Squirrel (50pts): 1`);
    console.log('');
    console.log('=== PHASE 4: STATS & PROGRESSION ===');
    console.log(`😊 Happiness: ${stats.happiness}/${stats.maxHappiness} (affects movement speed)`);
    console.log(`⚡ Energy: ${stats.energy}/${stats.maxEnergy} (constant drain, refill with bowls)`);
    console.log(`🎖️ Level: ${stats.level} | XP: ${stats.xp}/100`);
    console.log(`🏆 Achievements: ${achievementDefs.length} available`);
    console.log('');
    console.log('=== LEVEL LAYOUT ===');
    console.log(`🛏️ Checkpoints: ${checkpoints.length} | 🚧 Obstacles: ${obstacles.length}`);
    console.log('');
    console.log('=== PHASE 5: ADVANCED PLATFORMING ===');
    console.log('');
    console.log('🎮 Platform Mechanics:');
    console.log(`  ├─ 🪜 Ladders: ${ladders.length} climbable (UP/DOWN to climb)`);
    console.log(`  ├─ 🎢 Moving Platforms: ${movingPlatforms.length} (horizontal, vertical, circular)`);
    console.log(`  ├─ ⛰️  Slopes: ${slopes.length} angled ramps (slide on steep slopes)`);
    console.log(`  ├─ 💥 Falling Platforms: ${platforms.filter(p => p.type === 'falling').length} (crumble when stepped on)`);
    console.log(`  └─ 🏀 Bouncy Platforms: ${platforms.filter(p => p.type === 'bouncy').length} (spring jump 2x force)`);
    console.log('');
    console.log('🌍 Environmental Features:');
    console.log(`  ├─ 🗺️  Themed Zones: ${zones.length} (Sunny Park, Playground, Forest, Summit)`);
    console.log(`  ├─ 🏔️  Parallax Layers: ${backgroundLayers.length} background + ${foregroundLayers.length} foreground`);
    console.log(`  ├─ 🌸 Flower Decorations: ${flowerSprites.length} variants with depth effect`);
    console.log(`  ├─ 💧 Water Zones: ${waterZones.length} with swimming mechanics`);
    console.log(`  └─ 🔧 Interactive Elements: ${interactiveElements.length} (switches, doors, bridges)`);
    console.log('');
    console.log('🧗 Advanced Movement:');
    console.log('  ├─ Wall Jump: Slide on walls, jump to change direction');
    console.log('  ├─ Swimming: Hold JUMP/UP in water to swim upward');
    console.log('  └─ Ladder Climbing: Press UP to attach, SPACE to jump off');
    console.log('');
    console.log('=== CONTROLS ===');
    console.log('Movement: Arrow Keys / WASD');
    console.log('Jump: Space (also swim up in water)');
    console.log('Climb: UP/W (ladders) | DOWN/S (descend)');
    console.log('Interact: UP/W (switches, levers)');
    console.log('Toggle Frames: T (switch 1-16 ↔ 17-20)');
    console.log('Collection Tracker: C');
    console.log('');

    // Setup canvas responsive sizing
    resizeCanvas();
    window.addEventListener('resize', resizeCanvas);

    // Create level
    createLevel();

    // Set player starting position (on starting platform, which is platforms[1])
    player.x = 150;
    player.y = platforms[1].y - player.height - 2; // Place ON TOP of starting platform (not ground)

    // Setup input
    setupInput();

    // Start game
    gameRunning = true;
    lastTime = performance.now();
    gameLoop(lastTime);
}

// ===== CANVAS RESIZE (maintain aspect ratio) =====
function resizeCanvas() {
    const windowWidth = window.innerWidth;
    const windowHeight = window.innerHeight;
    const aspectRatio = CANVAS_WIDTH / CANVAS_HEIGHT;
    const windowAspect = windowWidth / windowHeight;

    let width, height;

    if (windowAspect > aspectRatio) {
        height = windowHeight;
        width = height * aspectRatio;
    } else {
        width = windowWidth;
        height = width / aspectRatio;
    }

    canvas.style.width = `${width}px`;
    canvas.style.height = `${height}px`;
}

// ===== LEVEL CREATION =====
function createLevel() {
    // Ground (solid platform) - extended to cover sky zone
    platforms.push({
        x: 0, y: GROUND_Y, width: CANVAS_WIDTH * 4, height: 100,
        color: '#8BC34A', type: 'solid'
    });

    // Starting platform
    platforms.push({
        x: 50, y: 500, width: 300, height: 30,
        color: '#8BC34A', type: 'solid'
    });

    // Staircase platforms (one-way - can jump through from below)
    platforms.push({
        x: 400, y: 450, width: 200, height: 30,
        color: '#7CB342', type: 'one-way'
    });
    platforms.push({
        x: 650, y: 400, width: 200, height: 30,
        color: '#7CB342', type: 'one-way'
    });
    platforms.push({
        x: 900, y: 350, width: 200, height: 30,
        color: '#7CB342', type: 'one-way'
    });

    // High platform (one-way)
    platforms.push({
        x: 700, y: 250, width: 300, height: 30,
        color: '#7CB342', type: 'one-way'
    });

    // Right side platforms (one-way)
    platforms.push({
        x: 1200, y: 400, width: 250, height: 30,
        color: '#7CB342', type: 'one-way'
    });
    platforms.push({
        x: 1500, y: 500, width: 200, height: 30,
        color: '#7CB342', type: 'one-way'
    });

    // Floating platforms (one-way)
    platforms.push({
        x: 1750, y: 350, width: 200, height: 30,
        color: '#7CB342', type: 'one-way'
    });
    platforms.push({
        x: 2000, y: 280, width: 200, height: 30,
        color: '#7CB342', type: 'one-way'
    });

    // Falling platforms (Phase 5 - crumble when stepped on)
    platforms.push({
        x: 1200, y: 250, width: 100, height: 20,
        color: '#D7CCC8', type: 'falling',
        state: 'stable', // stable, shaking, falling, respawning
        fallTimer: 0,
        fallDelay: 0.5, // Time before falling
        respawnTime: 3, // Time to respawn
        originalY: 250,
        fallVelocity: 0
    });
    platforms.push({
        x: 1650, y: 200, width: 80, height: 20,
        color: '#D7CCC8', type: 'falling',
        state: 'stable',
        fallTimer: 0,
        fallDelay: 0.5,
        respawnTime: 3,
        originalY: 200,
        fallVelocity: 0
    });
    platforms.push({
        x: 2400, y: 350, width: 90, height: 20,
        color: '#D7CCC8', type: 'falling',
        state: 'stable',
        fallTimer: 0,
        fallDelay: 0.5,
        respawnTime: 3,
        originalY: 350,
        fallVelocity: 0
    });

    // Bouncy platforms (Phase 5 - spring jump)
    platforms.push({
        x: 800, y: 500, width: 100, height: 20,
        color: '#FF9800', type: 'bouncy',
        bounceForce: -1200, // Extra strong jump
        squish: 0 // Visual squish effect
    });
    platforms.push({
        x: 1550, y: 450, width: 80, height: 20,
        color: '#FF9800', type: 'bouncy',
        bounceForce: -1200,
        squish: 0
    });
    platforms.push({
        x: 2700, y: 400, width: 100, height: 20,
        color: '#FF9800', type: 'bouncy',
        bounceForce: -1200,
        squish: 0
    });

    // Rainbow platforms (Phase 6 - colorful, extra bouncy)
    platforms.push({
        x: 1350, y: 280, width: 120, height: 18,
        type: 'rainbow',
        bounceForce: -1000,
        rainbowOffset: 0 // For animation
    });
    platforms.push({
        x: 2100, y: 320, width: 100, height: 18,
        type: 'rainbow',
        bounceForce: -1000,
        rainbowOffset: 0
    });
    platforms.push({
        x: 2950, y: 250, width: 110, height: 18,
        type: 'rainbow',
        bounceForce: -1000,
        rainbowOffset: 0
    });

    // Stairway to sky - platforms leading to ladder
    platforms.push({
        x: 3300, y: GROUND_Y - 50, width: 120, height: 25,
        color: '#FFF9C4', type: 'solid'
    });
    platforms.push({
        x: 3450, y: GROUND_Y - 80, width: 110, height: 25,
        color: '#FFF59D', type: 'solid'
    });
    platforms.push({
        x: 3600, y: GROUND_Y - 110, width: 150, height: 30,
        color: '#FFD700', type: 'solid' // Golden platform at ladder base
    });

    // ===== SKY LEVEL PLATFORMS (ABOVE - Vertical transition!) =====

    // Landing platform at top of ladder
    platforms.push({
        x: 3650, y: 120, width: 150, height: 25,
        color: '#E3F2FD', type: 'solid'
    });

    // Sky platforms extending right from ladder
    platforms.push({
        x: 3850, y: 140, width: 140, height: 22,
        color: '#BBDEFB', type: 'one-way'
    });
    platforms.push({
        x: 4030, y: 180, width: 130, height: 22,
        color: '#BBDEFB', type: 'one-way'
    });
    platforms.push({
        x: 4200, y: 150, width: 120, height: 22,
        color: '#BBDEFB', type: 'one-way'
    });

    // Upper sky platforms
    platforms.push({
        x: 3900, y: 240, width: 110, height: 22,
        color: '#E3F2FD', type: 'one-way'
    });
    platforms.push({
        x: 4100, y: 280, width: 120, height: 22,
        color: '#E3F2FD', type: 'one-way'
    });
    platforms.push({
        x: 4300, y: 320, width: 130, height: 22,
        color: '#E3F2FD', type: 'one-way'
    });

    // Cloud platforms (bouncy white clouds)
    platforms.push({
        x: 3750, y: 200, width: 90, height: 18,
        color: '#FFFFFF', type: 'bouncy',
        bounceForce: -1100,
        squish: 0
    });
    platforms.push({
        x: 4000, y: 100, width: 85, height: 18,
        color: '#FFFFFF', type: 'bouncy',
        bounceForce: -1100,
        squish: 0
    });
    platforms.push({
        x: 4250, y: 220, width: 80, height: 18,
        color: '#FFFFFF', type: 'bouncy',
        bounceForce: -1100,
        squish: 0
    });

    // Rainbow platforms in highest sky
    platforms.push({
        x: 3850, y: 80, width: 120, height: 18,
        type: 'rainbow',
        bounceForce: -1200,
        rainbowOffset: 0
    });
    platforms.push({
        x: 4150, y: 120, width: 100, height: 18,
        type: 'rainbow',
        bounceForce: -1200,
        rainbowOffset: 0
    });
    platforms.push({
        x: 4380, y: 160, width: 110, height: 18,
        type: 'rainbow',
        bounceForce: -1200,
        rainbowOffset: 0
    });

    // ==== SKY LEVEL - FLOATING ISLANDS! ====

    // LEFT ENTRANCE ISLAND (Large starting platform)
    platforms.push({
        x: 3850, y: 420, width: 200, height: 35,
        type: 'island'
    });

    // LOWER LEFT FLOATING ISLAND
    platforms.push({
        x: 3600, y: 360, width: 160, height: 30,
        type: 'island'
    });

    // LOWER CENTER ISLAND (Big main platform)
    platforms.push({
        x: 4050, y: 400, width: 220, height: 35,
        type: 'island'
    });

    // LOWER RIGHT ISLAND
    platforms.push({
        x: 4400, y: 390, width: 150, height: 30,
        type: 'island'
    });

    // MID-LEFT ISLAND (with tree)
    platforms.push({
        x: 3700, y: 280, width: 180, height: 32,
        type: 'island',
        hasTree: true
    });

    // MID-CENTER SMALL ISLAND
    platforms.push({
        x: 4000, y: 300, width: 130, height: 28,
        type: 'island'
    });

    // MID-RIGHT ISLAND (with pillar)
    platforms.push({
        x: 4350, y: 290, width: 170, height: 32,
        type: 'island',
        hasPillar: true
    });

    // UPPER-LEFT ISLAND
    platforms.push({
        x: 3650, y: 200, width: 140, height: 28,
        type: 'island'
    });

    // UPPER-CENTER ISLAND (Large platform)
    platforms.push({
        x: 3950, y: 210, width: 200, height: 32,
        type: 'island',
        hasTree: true
    });

    // UPPER-RIGHT ISLAND
    platforms.push({
        x: 4300, y: 190, width: 150, height: 30,
        type: 'island'
    });

    // TOP-LEFT ISLAND (Bonus area)
    platforms.push({
        x: 3750, y: 120, width: 160, height: 28,
        type: 'island'
    });

    // TOP-RIGHT ISLAND (Highest reward)
    platforms.push({
        x: 4200, y: 100, width: 180, height: 30,
        type: 'island',
        hasPillar: true
    });

    // Small cloud platforms (stepping stones)
    platforms.push({
        x: 3920, y: 350, width: 70, height: 15,
        color: '#FFFFFF', type: 'bouncy',
        bounceForce: -900,
        squish: 0
    });
    platforms.push({
        x: 4280, y: 340, width: 65, height: 15,
        color: '#FFFFFF', type: 'bouncy',
        bounceForce: -900,
        squish: 0
    });
    platforms.push({
        x: 3850, y: 250, width: 70, height: 15,
        color: '#FFFFFF', type: 'bouncy',
        bounceForce: -950,
        squish: 0
    });
    platforms.push({
        x: 4180, y: 240, width: 65, height: 15,
        color: '#FFFFFF', type: 'bouncy',
        bounceForce: -950,
        squish: 0
    });
    platforms.push({
        x: 3900, y: 160, width: 70, height: 15,
        color: '#FFFFFF', type: 'bouncy',
        bounceForce: -1000,
        squish: 0
    });
    platforms.push({
        x: 4500, y: 150, width: 70, height: 15,
        color: '#FFFFFF', type: 'bouncy',
        bounceForce: -1000,
        squish: 0
    });

    // Collectibles (stars)
    addCollectible(200, 450);
    addCollectible(500, 400);
    addCollectible(750, 350);
    addCollectible(1000, 300);
    addCollectible(800, 200);
    addCollectible(1300, 350);
    addCollectible(1600, 450);
    addCollectible(1850, 300);
    addCollectible(2100, 230);
    addCollectible(500, GROUND_Y - 60);
    addCollectible(1000, GROUND_Y - 60);
    addCollectible(1500, GROUND_Y - 60);

    // Add special collectibles (Phase 2: balls and frisbees)
    addSpecialCollectibles();

    // Create clouds (background parallax scenery)
    createClouds();

    // Create trees (platform decorations)
    createTrees();

    // Create obstacles (Phase 2)
    createObstacles();

    // Create checkpoints (Phase 2)
    createCheckpoints();

    // Create NPCs (Phase 3)
    createNPCs();

    // Create squirrel (Phase 3)
    createSquirrel();

    // Create moving platforms (Phase 5)
    createMovingPlatforms();

    // Create ladders (Phase 5)
    createLadders();

    // Create slopes (Phase 5)
    createSlopes();

    // Create parallax backgrounds and foregrounds (Phase 5)
    createParallaxLayers();

    // Create themed zones (Phase 5)
    createZones();

    // Create interactive elements (Phase 5)
    createInteractiveElements();

    // Create water zones (Phase 5)
    createWaterZones();
}

function addCollectible(x, y, type = 'bone') {
    const collectible = {
        x: x,
        y: y,
        width: 40,
        height: 40,
        collected: false,
        floatTime: Math.random() * Math.PI * 2,
        initialY: y,
        type: type, // 'bone', 'ball', 'frisbee', 'water', 'food'
        rotationSpeed: 0.5 + Math.random() * 1.5
    };

    // Type-specific properties
    if (type === 'bone') {
        collectible.boneVariant = Math.floor(Math.random() * totalBones);
        collectible.points = 10;
        collectible.happinessGain = 10;
        collectible.xpGain = 10;
    } else if (type === 'ball') {
        collectible.points = 20;
        collectible.happinessGain = 15;
        collectible.xpGain = 20;
        collectible.bounceVelocity = 0;
        collectible.bounceTime = 0;
    } else if (type === 'frisbee') {
        collectible.points = 30;
        collectible.happinessGain = 20;
        collectible.xpGain = 30;
        collectible.arcTime = 0;
        collectible.arcHeight = 30;
    } else if (type === 'water') {
        collectible.points = 0;
        collectible.energyGain = 30;
        collectible.happinessGain = 5;
    } else if (type === 'food') {
        collectible.points = 0;
        collectible.energyGain = 50;
        collectible.happinessGain = 10;
    }

    collectibles.push(collectible);
}

function createClouds() {
    // Create 5 clouds at different positions and heights
    // Clouds use parallax scrolling (move slower than camera)
    clouds.push({
        x: 300,
        y: 100,
        variant: 0, // cloud_1_
        speed: 0.3, // Parallax factor (30% of camera speed)
        width: 120,
        height: 60
    });

    clouds.push({
        x: 800,
        y: 150,
        variant: 1, // cloud_2_
        speed: 0.2,
        width: 140,
        height: 70
    });

    clouds.push({
        x: 1400,
        y: 80,
        variant: 2, // cloud_3_
        speed: 0.25,
        width: 100,
        height: 50
    });

    clouds.push({
        x: 2000,
        y: 120,
        variant: 0, // cloud_1_ again
        speed: 0.35,
        width: 120,
        height: 60
    });

    clouds.push({
        x: 2600,
        y: 90,
        variant: 1, // cloud_2_ again
        speed: 0.28,
        width: 140,
        height: 70
    });

    // Sky level clouds (well-spaced decorative clouds)
    clouds.push({
        x: 3650,
        y: 200,
        variant: 2,
        speed: 0.25,
        width: 130,
        height: 65
    });

    clouds.push({
        x: 3950,
        y: 380,
        variant: 0,
        speed: 0.3,
        width: 150,
        height: 75
    });

    clouds.push({
        x: 4250,
        y: 300,
        variant: 1,
        speed: 0.22,
        width: 140,
        height: 70
    });

    clouds.push({
        x: 4500,
        y: 420,
        variant: 2,
        speed: 0.28,
        width: 120,
        height: 60
    });

    clouds.push({
        x: 4600,
        y: 180,
        variant: 0,
        speed: 0.26,
        width: 135,
        height: 68
    });
}

function createTrees() {
    // Place trees on specific platforms as decorations
    // Platform indices: 5 = high platform, 0 = ground

    // Large tree with squirrel on high platform
    trees.push({
        x: 750,
        y: platforms[5].y - 150, // 150px above platform
        variant: 0, // tree_1_ (has squirrel!)
        width: 120,
        height: 150
    });

    // Small tree on ground - moved away from crowded area
    trees.push({
        x: 1700,
        y: GROUND_Y - 80,
        variant: 1, // tree_2_
        width: 60,
        height: 80
    });

    // Another large tree further in level
    trees.push({
        x: 2900,
        y: GROUND_Y - 150,
        variant: 0, // tree_1_
        width: 120,
        height: 150
    });
}

function createObstacles() {
    // Fire hydrants - decorative (removed solid collision, too annoying)
    // Players can now walk through them

    // Decorative hydrant 1
    obstacles.push({
        type: 'hydrant',
        x: 600,
        y: GROUND_Y - 60,
        width: 40,
        height: 60,
        solid: false // Not blocking anymore
    });

    // Decorative hydrant 2
    obstacles.push({
        type: 'hydrant',
        x: 2500,
        y: GROUND_Y - 60,
        width: 40,
        height: 60,
        solid: false // Not blocking anymore
    });
}

function createCheckpoints() {
    // Dog beds - save spawn points

    // Starting checkpoint (default)
    checkpoints.push({
        x: 100,
        y: platforms[1].y - 50, // On starting platform
        width: 60,
        height: 50,
        activated: true // Start here
    });

    // Mid-level checkpoint
    checkpoints.push({
        x: 1000,
        y: GROUND_Y - 50,
        width: 60,
        height: 50,
        activated: false
    });

    // Late-level checkpoint
    checkpoints.push({
        x: 2000,
        y: GROUND_Y - 50,
        width: 60,
        height: 50,
        activated: false
    });

    // Sky level checkpoint (at base of sky ladder)
    checkpoints.push({
        x: 3650,
        y: GROUND_Y - 50,
        width: 60,
        height: 50,
        activated: false
    });

    // Sky level checkpoint (at top of sky)
    checkpoints.push({
        x: 3650,
        y: 70,  // High in the sky!
        width: 60,
        height: 50,
        activated: false
    });
}

function addSpecialCollectibles() {
    // Add balls (20 points each) - 4 total
    addCollectible(700, 400, 'ball');  // Near first hydrant
    addCollectible(1350, 350, 'ball'); // Near second hydrant
    addCollectible(1800, GROUND_Y - 100, 'ball');
    addCollectible(2250, 250, 'ball'); // High platform

    // Add frisbees (30 points each) - 2 total (rare!)
    addCollectible(900, 200, 'frisbee');  // High platform risk
    addCollectible(2150, 150, 'frisbee'); // Late game challenge

    // Add water bowls (Phase 4) - +30 energy
    addCollectible(800, GROUND_Y - 60, 'water');
    addCollectible(2400, GROUND_Y - 60, 'water');

    // Add food bowls (Phase 4) - +50 energy
    addCollectible(1000, GROUND_Y - 60, 'food');
    addCollectible(1800, GROUND_Y - 60, 'food');

    // Add apples (15 points, +15 happiness) - New!
    addCollectible(600, 350, 'apple');
    addCollectible(1100, 250, 'apple');
    addCollectible(1600, 400, 'apple');
    addCollectible(2300, 300, 'apple');
    addCollectible(2800, GROUND_Y - 100, 'apple');

    // Add lightning bolts (50 points, bonus multiplier) - New!
    addCollectible(1200, 180, 'lightning');
    addCollectible(2400, 200, 'lightning');

    // ===== SKY LEVEL COLLECTIBLES (Scattered like reference!) =====

    // LEFT ENTRANCE ISLAND collectibles
    addCollectible(3900, 370, 'bone');
    addCollectible(3950, 370, 'bone');
    addCollectible(4000, 370, 'bone');

    // LOWER LEFT ISLAND
    addCollectible(3650, 320, 'apple');
    addCollectible(3700, 320, 'bone');

    // LOWER CENTER ISLAND (Big platform - multiple coins)
    addCollectible(4100, 350, 'bone');
    addCollectible(4150, 350, 'bone');
    addCollectible(4200, 350, 'bone');
    addCollectible(4180, 350, 'apple');

    // LOWER RIGHT ISLAND
    addCollectible(4450, 340, 'ball');
    addCollectible(4500, 340, 'bone');

    // Between lower and mid platforms (in air)
    addCollectible(3780, 335, 'bone');
    addCollectible(4320, 335, 'bone');

    // MID-LEFT ISLAND (with tree)
    addCollectible(3750, 240, 'apple');
    addCollectible(3800, 240, 'bone');
    addCollectible(3850, 240, 'lightning');

    // MID-CENTER ISLAND
    addCollectible(4050, 260, 'bone');
    addCollectible(4080, 260, 'bone');

    // MID-RIGHT ISLAND (with pillar)
    addCollectible(4400, 250, 'ball');
    addCollectible(4450, 250, 'bone');
    addCollectible(4500, 250, 'apple');

    // Between mid and upper platforms (in air)
    addCollectible(3880, 230, 'bone');
    addCollectible(4200, 220, 'bone');

    // UPPER-LEFT ISLAND
    addCollectible(3700, 160, 'apple');
    addCollectible(3750, 160, 'bone');

    // UPPER-CENTER ISLAND (with tree - multiple rewards)
    addCollectible(4000, 170, 'bone');
    addCollectible(4050, 170, 'bone');
    addCollectible(4100, 170, 'lightning');
    addCollectible(4080, 170, 'apple');

    // UPPER-RIGHT ISLAND
    addCollectible(4350, 150, 'ball');
    addCollectible(4400, 150, 'bone');

    // TOP-LEFT ISLAND (Bonus area - big rewards!)
    addCollectible(3800, 80, 'lightning');
    addCollectible(3850, 80, 'frisbee');
    addCollectible(3780, 80, 'apple');

    // TOP-RIGHT ISLAND (Highest reward - jackpot!)
    addCollectible(4250, 60, 'lightning');
    addCollectible(4300, 60, 'frisbee');
    addCollectible(4350, 60, 'lightning');

    // On cloud platforms (bonus for skilled jumps)
    addCollectible(3920, 310, 'bone');
    addCollectible(4280, 300, 'bone');
    addCollectible(3850, 210, 'apple');
    addCollectible(4180, 200, 'apple');
    addCollectible(3900, 120, 'lightning');
    addCollectible(4500, 110, 'lightning');
}

function createNPCs() {
    // Phase 3: Create 6 friendly NPC dogs
    // 4 static, 2 patrol

    // Buddy - Static on starting platform
    npcs.push({
        name: 'Buddy',
        sprite: npcDogSprites[0], // dog_1_
        x: 250,
        y: platforms[1].y - 60,
        width: 60,
        height: 60,
        type: 'static',
        greeted: false,
        barkTime: 0,
        facingRight: true
    });

    // Max - Patrols ground level
    npcs.push({
        name: 'Max',
        sprite: npcDogSprites[1], // dog_2_
        x: 500,
        y: GROUND_Y - 60,
        width: 60,
        height: 60,
        type: 'patrol',
        minX: 400,
        maxX: 700,
        speed: 100,
        direction: 1,
        greeted: false,
        barkTime: 0,
        facingRight: true
    });

    // Luna - Static on high platform
    npcs.push({
        name: 'Luna',
        sprite: npcDogSprites[2], // dog_3_
        x: 850,
        y: platforms[5].y - 60, // High platform
        width: 60,
        height: 60,
        type: 'static',
        greeted: false,
        barkTime: 0,
        facingRight: false // Facing left
    });

    // Rocky - Patrols mid-level platforms
    npcs.push({
        name: 'Rocky',
        sprite: npcDogSprites[3], // dog_4_
        x: 1200,
        y: platforms[6].y - 60, // Platform[6]
        width: 60,
        height: 60,
        type: 'patrol',
        minX: 1200,
        maxX: 1400,
        speed: 80,
        direction: 1,
        greeted: false,
        barkTime: 0,
        facingRight: true
    });

    // Daisy - Static near small tree
    npcs.push({
        name: 'Daisy',
        sprite: npcDogSprites[4], // dog_5_
        x: 1150,
        y: GROUND_Y - 60,
        width: 60,
        height: 60,
        type: 'static',
        greeted: false,
        barkTime: 0,
        facingRight: true
    });

    // Scout - Static near water zone
    npcs.push({
        name: 'Scout',
        sprite: npcDogSprites[5], // dog_6_
        x: 1600,
        y: GROUND_Y - 60,
        width: 60,
        height: 60,
        type: 'static',
        greeted: false,
        barkTime: 0,
        facingRight: false
    });

    // ===== SKY LEVEL NPCs (Birds on Islands!) =====

    // Tweety - On lower left island
    npcs.push({
        name: 'Tweety',
        sprite: npcDogSprites[0],
        x: 3660,
        y: 320,
        width: 40,
        height: 40,
        type: 'bird',
        greeted: false,
        barkTime: 0,
        facingRight: true,
        isBird: true
    });

    // Skyler - Flying patrol between mid islands
    npcs.push({
        name: 'Skyler',
        sprite: npcDogSprites[1],
        x: 4000,
        y: 260,
        width: 40,
        height: 40,
        type: 'flying-patrol',
        minX: 3750,
        maxX: 4400,
        minY: 230,
        maxY: 300,
        speed: 70,
        direction: 1,
        greeted: false,
        barkTime: 0,
        facingRight: true,
        isBird: true
    });

    // Cirrus - On upper-center island (with tree)
    npcs.push({
        name: 'Cirrus',
        sprite: npcDogSprites[3],
        x: 4100,
        y: 170,
        width: 40,
        height: 40,
        type: 'bird',
        greeted: false,
        barkTime: 0,
        facingRight: false,
        isBird: true
    });

    // Cloud - On top island (bonus reward guardian)
    npcs.push({
        name: 'Cloud',
        sprite: npcDogSprites[2],
        x: 3810,
        y: 80,
        width: 40,
        height: 40,
        type: 'bird',
        greeted: false,
        barkTime: 0,
        facingRight: true,
        isBird: true
    });
}

function createSquirrel() {
    // Squirrel chase mini-game
    squirrel = {
        x: 750,
        y: platforms[5].y - 180, // In tree at platform 5
        width: 25,
        height: 25,
        state: 'idle', // idle, alerted, running, caught
        targetX: 750,
        speed: 250,
        caught: false,
        respawnTimer: 0,
        alertDistance: 150,
        chasePhase: 0 // 0=tree, 1=jump, 2=run1, 3=run2, 4=caught
    };
}

function createMovingPlatforms() {
    // Phase 5: Create moving platforms (horizontal, vertical, circular)

    // Horizontal moving platforms (left-right patrol)
    movingPlatforms.push({
        x: 1100,
        y: 300,
        width: 120,
        height: 20,
        color: '#81C784',
        type: 'horizontal',
        minX: 1000,
        maxX: 1300,
        speed: 80,
        direction: 1,
        velocityX: 0,
        velocityY: 0
    });

    movingPlatforms.push({
        x: 1600,
        y: 350,
        width: 100,
        height: 20,
        color: '#81C784',
        type: 'horizontal',
        minX: 1550,
        maxX: 1850,
        speed: 120,
        direction: 1,
        velocityX: 0,
        velocityY: 0
    });

    // Vertical moving platforms (elevators)
    movingPlatforms.push({
        x: 1400,
        y: 450,
        width: 100,
        height: 20,
        color: '#64B5F6',
        type: 'vertical',
        minY: 300,
        maxY: 550,
        speed: 100,
        direction: 1,
        velocityX: 0,
        velocityY: 0
    });

    movingPlatforms.push({
        x: 2200,
        y: 400,
        width: 120,
        height: 20,
        color: '#64B5F6',
        type: 'vertical',
        minY: 250,
        maxY: 500,
        speed: 70,
        direction: -1,
        velocityX: 0,
        velocityY: 0
    });

    // Circular moving platform (orbits a center point)
    movingPlatforms.push({
        x: 2500,
        y: 400,
        width: 100,
        height: 20,
        color: '#FFB74D',
        type: 'circular',
        centerX: 2550,
        centerY: 400,
        radius: 100,
        angle: 0,
        angularSpeed: 1.5, // radians per second
        velocityX: 0,
        velocityY: 0
    });

    // ===== SKY LEVEL MOVING CLOUD PLATFORMS =====

    // Horizontal drifting cloud - between lower islands
    movingPlatforms.push({
        x: 3750,
        y: 380,
        width: 85,
        height: 16,
        color: '#FFFFFF',
        type: 'horizontal',
        minX: 3700,
        maxX: 3950,
        speed: 50,
        direction: 1,
        velocityX: 0,
        velocityY: 0
    });

    // Vertical floating cloud - mid-right area
    movingPlatforms.push({
        x: 4550,
        y: 240,
        width: 80,
        height: 16,
        color: '#FFFFFF',
        type: 'vertical',
        minY: 220,
        maxY: 280,
        speed: 45,
        direction: 1,
        velocityX: 0,
        velocityY: 0
    });

    // Horizontal cloud - upper sky area
    movingPlatforms.push({
        x: 3850,
        y: 160,
        width: 75,
        height: 16,
        color: '#FFFFFF',
        type: 'horizontal',
        minX: 3800,
        maxX: 4100,
        speed: 55,
        direction: 1,
        velocityX: 0,
        velocityY: 0
    });

    // Circular orbiting cloud - around mid island
    movingPlatforms.push({
        x: 4150,
        y: 310,
        width: 80,
        height: 16,
        color: '#E3F2FD',
        type: 'circular',
        centerX: 4200,
        centerY: 320,
        radius: 65,
        angle: 0,
        angularSpeed: 0.8,
        velocityX: 0,
        velocityY: 0
    });
}

function createLadders() {
    // Phase 5: Create climbable ladders

    // Ladder to high platform (zone 1)
    ladders.push({
        x: 750,
        y: 250,
        width: 30,
        height: 250, // Reaches from y=250 to y=500
        color: '#8D6E63'
    });

    // Tall ladder in mid-section (zone 2)
    ladders.push({
        x: 1350,
        y: 200,
        width: 30,
        height: 420, // Extra tall
        color: '#8D6E63'
    });

    // Ladder near moving platform (zone 3)
    ladders.push({
        x: 2150,
        y: 350,
        width: 30,
        height: 270,
        color: '#8D6E63'
    });

    // MASSIVE SKY LADDER - Transition to Sky Level!
    ladders.push({
        x: 3700,
        y: 100,  // Top of sky level
        width: 35,
        height: 520, // Goes all the way down to ground
        color: '#FFD700' // Golden ladder to the sky!
    });

    // Sky island connecting ladders
    // Ladder from lower left to mid-left island
    ladders.push({
        x: 3720,
        y: 280,
        width: 30,
        height: 80,
        color: '#B0BEC5'
    });

    // Ladder from mid-center to upper-center island
    ladders.push({
        x: 4060,
        y: 210,
        width: 30,
        height: 90,
        color: '#B0BEC5'
    });

    // Ladder from upper-center to top island
    ladders.push({
        x: 3970,
        y: 120,
        width: 30,
        height: 90,
        color: '#B0BEC5'
    });

    // Small ladder on right side (lower to mid)
    ladders.push({
        x: 4430,
        y: 290,
        width: 30,
        height: 100,
        color: '#B0BEC5'
    });

    // Ladder to top-right island
    ladders.push({
        x: 4320,
        y: 100,
        width: 30,
        height: 90,
        color: '#B0BEC5'
    });
}

function createSlopes() {
    // Phase 5: Create angled ramps/slopes
    // Slopes are defined by start point, width, and angle
    // Positive angle = slopes upward to the right
    // Negative angle = slopes downward to the right

    // Gentle upward ramp (zone 1)
    slopes.push({
        x: 300,
        y: 550,
        width: 200,
        height: 60, // Visual height
        angle: 15, // degrees
        direction: 'up',
        color: '#9CCC65',
        endY: 490 // Y position at the end of slope
    });

    // Small gentle ramp (zone 2) - moved and made smaller
    slopes.push({
        x: 200,
        y: 540,
        width: 100,
        height: 30,
        angle: -15,
        direction: 'down',
        color: '#AED581',
        endY: 565
    });

    // Steep upward climb (zone 3)
    slopes.push({
        x: 1900,
        y: 550,
        width: 180,
        height: 90,
        angle: 30,
        direction: 'up',
        color: '#8BC34A',
        endY: 460
    });

    // Long gentle slope (zone 4)
    slopes.push({
        x: 2600,
        y: 500,
        width: 250,
        height: 75,
        angle: 18,
        direction: 'up',
        color: '#9CCC65',
        endY: 425
    });
}

function createParallaxLayers() {
    // Phase 5: Create multi-layer parallax backgrounds and foregrounds

    // Background Layer 1: Far mountains (slowest parallax)
    backgroundLayers.push({
        type: 'mountains',
        parallaxSpeed: 0.1,
        yOffset: 420,
        color: '#7B68AA',
        mountains: [
            { x: 200, width: 300, height: 150 },
            { x: 600, width: 400, height: 180 },
            { x: 1200, width: 350, height: 160 },
            { x: 1800, width: 450, height: 200 },
            { x: 2400, width: 300, height: 140 },
            { x: 3000, width: 380, height: 170 }
        ]
    });

    // Background Layer 2: Distant trees (medium parallax)
    backgroundLayers.push({
        type: 'trees',
        parallaxSpeed: 0.25,
        yOffset: 480,
        positions: [500, 1100, 1700, 2300, 2900, 3500]
    });

    // Foreground Layer 1: Flowers (faster than camera - creates depth)
    // Place flowers at ground level across the map
    for (let x = 100; x < CANVAS_WIDTH * 3; x += 200) {
        foregroundLayers.push({
            type: 'flower',
            x: x + Math.random() * 100,
            y: GROUND_Y - 40,
            variant: Math.floor(Math.random() * 3),
            parallaxSpeed: 1.15, // Moves 15% faster than camera
            size: 30 + Math.random() * 20
        });
    }

    // Add more flowers scattered around platforms
    foregroundLayers.push(
        { type: 'flower', x: 350, y: 470, variant: 0, parallaxSpeed: 1.1, size: 35 },
        { type: 'flower', x: 720, y: 220, variant: 1, parallaxSpeed: 1.12, size: 40 },
        { type: 'flower', x: 1150, y: 370, variant: 2, parallaxSpeed: 1.08, size: 32 },
        { type: 'flower', x: 1450, y: 470, variant: 0, parallaxSpeed: 1.15, size: 38 },
        { type: 'flower', x: 1900, y: 420, variant: 1, parallaxSpeed: 1.1, size: 33 },
        { type: 'flower', x: 2300, y: 250, variant: 2, parallaxSpeed: 1.13, size: 36 }
    );
}

function createZones() {
    // Phase 5: Create 4 themed zones across the level
    // Each 960px wide (total 3840px level)

    // Zone 1: Sunny Park (0-960px) - TUTORIAL
    zones.push({
        name: 'Sunny Park',
        startX: 0,
        endX: 960,
        skyGradientTop: '#87CEEB',    // Bright blue
        skyGradientBottom: '#E0F6FF', // Light cyan
        mountainColor: '#9575CD',      // Purple-ish
        theme: 'park',
        ambientColor: 'rgba(255, 255, 200, 0.1)' // Warm yellow tint
    });

    // Zone 2: Playground (960-1920px) - CLIMBING
    zones.push({
        name: 'Playground',
        startX: 960,
        endX: 1920,
        skyGradientTop: '#87CEEB',    // Sky blue
        skyGradientBottom: '#E0F6FF', // Light cyan
        mountainColor: '#9575CD',      // Purple-ish
        theme: 'playground',
        ambientColor: 'rgba(255, 255, 200, 0.1)' // Warm tint
    });

    // Zone 3: Forest (1920-2880px) - VERTICAL
    zones.push({
        name: 'Misty Forest',
        startX: 1920,
        endX: 2880,
        skyGradientTop: '#78909C',    // Gray-blue
        skyGradientBottom: '#B0BEC5', // Light gray
        mountainColor: '#607D8B',      // Dark gray
        theme: 'forest',
        ambientColor: 'rgba(180, 200, 200, 0.15)' // Misty tint
    });

    // Zone 4: Summit (2880-3840px) - CHALLENGE
    zones.push({
        name: 'Cloud Summit',
        startX: 2880,
        endX: 3840,
        skyGradientTop: '#CE93D8',    // Purple sunset
        skyGradientBottom: '#F3E5F5', // Light purple
        mountainColor: '#BA68C8',      // Purple mountains
        theme: 'summit',
        ambientColor: 'rgba(255, 220, 255, 0.1)' // Purple tint
    });

    // Zone 5: Sky Realm (3500-4500px) - VERTICAL SKY LEVEL!
    zones.push({
        name: '☁️ Sky Realm ☁️',
        startX: 3500,
        endX: 4500,
        skyGradientTop: '#87CEEB',    // Bright sky blue
        skyGradientBottom: '#E0F6FF', // Very light blue
        mountainColor: '#FFFFFF',      // White clouds
        theme: 'sky',
        ambientColor: 'rgba(255, 255, 255, 0.15)' // Bright airy tint
    });
}

function createInteractiveElements() {
    // Phase 5: Create switches, doors, and bridges
    // (Removed confusing switch and door from zone 2)

    // Bridge 1 extends across gap in zone 3
    interactiveElements.push({
        type: 'bridge',
        x: 2100,
        y: 450,
        width: 200,
        height: 20,
        extended: false,
        linkedId: 'lever_1',
        color: '#8D6E63',
        retractedWidth: 50
    });

    // Lever to extend bridge
    interactiveElements.push({
        type: 'lever',
        x: 2050,
        y: GROUND_Y - 60,
        width: 25,
        height: 50,
        activated: false,
        linkedId: 'bridge_1',
        color: '#FFB74D'
    });

    // Door before sky area removed - free access to sky level!
}

function createWaterZones() {
    // Phase 5: Create water areas with swim mechanics

    // Water pool 1 in zone 2 (shallow) - repositioned
    waterZones.push({
        x: 1100,
        y: GROUND_Y - 100,
        width: 200,
        height: 100,
        color: 'rgba(100, 180, 255, 0.5)',
        buoyancy: 400, // Upward force in water
        drag: 0.7      // Movement multiplier
    });

    // Deep water in zone 3
    waterZones.push({
        x: 2200,
        y: GROUND_Y - 200,
        width: 300,
        height: 200,
        color: 'rgba(70, 150, 230, 0.6)',
        buoyancy: 500,
        drag: 0.6
    });

    // Water pit in zone 4 (requires swimming across)
    waterZones.push({
        x: 3000,
        y: GROUND_Y - 120,
        width: 350,
        height: 120,
        color: 'rgba(100, 180, 255, 0.5)',
        buoyancy: 450,
        drag: 0.65
    });
}

// ===== INPUT SETUP =====
function setupInput() {
    // Keyboard
    window.addEventListener('keydown', (e) => {
        if (e.code === 'ArrowLeft' || e.code === 'KeyA') keys.left = true;
        if (e.code === 'ArrowRight' || e.code === 'KeyD') keys.right = true;
        if (e.code === 'ArrowUp' || e.code === 'KeyW') keys.up = true; // Phase 5: Ladder climbing
        if (e.code === 'ArrowDown' || e.code === 'KeyS') keys.down = true; // Phase 5: Ladder climbing
        if ((e.code === 'Space' || e.code === 'ArrowUp' || e.code === 'KeyW') && !keys.jumpPressed) {
            keys.jump = true;
            keys.jumpPressed = true;
        }

        // Toggle between frame sets with 'T' key
        if (e.code === 'KeyT') {
            useFrameSet2 = !useFrameSet2;
            const frameSetText = useFrameSet2 ? '17-20' : '1-16';
            console.log(`🎨 Switched to Frame Set: ${frameSetText}`);

            // Update UI indicator
            const frameSetIndicator = document.getElementById('frameSet');
            if (frameSetIndicator) {
                frameSetIndicator.textContent = `Frames: ${frameSetText}`;
            }
        }

        // Toggle collection tracker with 'C' key
        if (e.code === 'KeyC') {
            showCollectionTracker = !showCollectionTracker;
            console.log(`📊 Collection Tracker: ${showCollectionTracker ? 'OPEN' : 'CLOSED'}`);
        }

        if (['ArrowUp', 'ArrowDown', 'ArrowLeft', 'ArrowRight', 'Space'].includes(e.code)) {
            e.preventDefault();
        }
    });

    window.addEventListener('keyup', (e) => {
        if (e.code === 'ArrowLeft' || e.code === 'KeyA') keys.left = false;
        if (e.code === 'ArrowRight' || e.code === 'KeyD') keys.right = false;
        if (e.code === 'ArrowUp' || e.code === 'KeyW') keys.up = false;
        if (e.code === 'ArrowDown' || e.code === 'KeyS') keys.down = false;
        if (e.code === 'Space' || e.code === 'ArrowUp' || e.code === 'KeyW') {
            keys.jumpPressed = false;
        }
    });

    // Touch controls
    const leftBtn = document.getElementById('leftBtn');
    const rightBtn = document.getElementById('rightBtn');
    const upBtn = document.getElementById('upBtn');
    const downBtn = document.getElementById('downBtn');
    const jumpBtn = document.getElementById('jumpBtn');

    function addTouchControl(btn, action) {
        const down = () => {
            if (action === 'left') keys.left = true;
            if (action === 'right') keys.right = true;
            if (action === 'up') {
                keys.up = true;
                if (!keys.jumpPressed) {
                    keys.jump = true;
                    keys.jumpPressed = true;
                }
            }
            if (action === 'down') keys.down = true;
            if (action === 'jump' && !keys.jumpPressed) {
                keys.jump = true;
                keys.jumpPressed = true;
            }
        };

        const up = () => {
            if (action === 'left') keys.left = false;
            if (action === 'right') keys.right = false;
            if (action === 'up') {
                keys.up = false;
                keys.jumpPressed = false;
            }
            if (action === 'down') keys.down = false;
            if (action === 'jump') keys.jumpPressed = false;
        };

        btn.addEventListener('touchstart', (e) => { e.preventDefault(); down(); });
        btn.addEventListener('touchend', (e) => { e.preventDefault(); up(); });
        btn.addEventListener('touchcancel', (e) => { e.preventDefault(); up(); });
        btn.addEventListener('mousedown', down);
        btn.addEventListener('mouseup', up);
        btn.addEventListener('mouseleave', up);
    }

    addTouchControl(leftBtn, 'left');
    addTouchControl(rightBtn, 'right');
    addTouchControl(upBtn, 'up');
    addTouchControl(downBtn, 'down');
    addTouchControl(jumpBtn, 'jump');

    // Make frameSet indicator tappable for mobile toggle
    const frameSetIndicator = document.getElementById('frameSet');
    if (frameSetIndicator) {
        const toggleFrames = () => {
            useFrameSet2 = !useFrameSet2;
            const frameSetText = useFrameSet2 ? '17-20' : '1-16';
            console.log(`🎨 Switched to Frame Set: ${frameSetText}`);
            frameSetIndicator.textContent = `Frames: ${frameSetText}`;
        };

        frameSetIndicator.addEventListener('click', toggleFrames);
        frameSetIndicator.addEventListener('touchend', (e) => {
            e.preventDefault();
            toggleFrames();
        });
    }
}

// ===== GAME LOOP =====
function gameLoop(currentTime) {
    if (!gameRunning) return;

    requestAnimationFrame(gameLoop);

    const deltaTime = Math.min((currentTime - lastTime) / 1000, 0.1);
    lastTime = currentTime;

    update(deltaTime);
    updateUI();
    render();
}

function updateUI() {
    // Update lives display
    const livesEl = document.getElementById('lives');
    if (livesEl) livesEl.textContent = `❤️ x${stats.lives}`;

    // Update level display
    const levelEl = document.getElementById('level');
    if (levelEl) levelEl.textContent = `L${stats.level}`;

    // Update score display
    const scoreEl = document.getElementById('score');
    if (scoreEl) scoreEl.textContent = `Score: ${score}`;

    // Update bonus score display
    const bonusEl = document.getElementById('bonusScore');
    if (bonusEl) bonusEl.textContent = `Bonus: ${stats.bonusScore}`;
}

// ===== UPDATE =====
function update(deltaTime) {
    // Calculate stat-based speed modifier (Phase 4)
    const happinessModifier = 0.7 + (stats.happiness / stats.maxHappiness) * 0.3; // 70%-100%
    const energyModifier = 0.5 + (stats.energy / stats.maxEnergy) * 0.5; // 50%-100%
    const levelSpeedBonus = stats.level >= 4 ? 1.05 : 1.0; // +5% at level 4+
    const powerupMultiplier = (powerup.active && powerup.type === 'speed') ? powerup.speedMultiplier : 1.0;
    const totalSpeedModifier = happinessModifier * energyModifier * levelSpeedBonus * powerupMultiplier;

    // Horizontal movement
    player.velocityX = 0;

    if (keys.left) {
        player.velocityX = -PLAYER.speed * totalSpeedModifier;
        player.facingRight = false;
    }

    if (keys.right) {
        player.velocityX = PLAYER.speed * totalSpeedModifier;
        player.facingRight = true;
    }

    // Jumping
    if (keys.jump && player.onGround) {
        player.velocityY = PLAYER.jumpForce;
        player.onGround = false;
    }

    keys.jump = false; // Reset jump (single press)

    // Apply gravity (disabled when climbing)
    if (!player.climbing) {
        player.velocityY += PLAYER.gravity * deltaTime;

        // Cap fall speed
        if (player.velocityY > PLAYER.maxFallSpeed) {
            player.velocityY = PLAYER.maxFallSpeed;
        }
    }

    // Update position
    player.x += player.velocityX * deltaTime;
    player.y += player.velocityY * deltaTime;

    // Platform collision
    player.onGround = false;

    for (const platform of platforms) {
        // Skip falling platforms that are currently falling or respawning
        if (platform.type === 'falling' && (platform.state === 'falling' || platform.state === 'respawning')) {
            continue;
        }

        if (checkCollision(player, platform)) {
            // One-way platforms: only land if coming from above
            if (platform.type === 'one-way') {
                if (player.velocityY > 0 &&
                    player.y + player.height - player.velocityY * deltaTime <= platform.y + 5) {
                    player.y = platform.y - player.height;
                    player.velocityY = 0;
                    player.onGround = true;
                    player.currentPlatform = platform;
                }
            }
            // Falling platforms: trigger fall when landed on
            else if (platform.type === 'falling') {
                if (player.velocityY > 0 &&
                    player.y + player.height - player.velocityY * deltaTime <= platform.y + 5) {
                    player.y = platform.y - player.height;
                    player.velocityY = 0;
                    player.onGround = true;
                    player.currentPlatform = platform;

                    // Trigger falling if stable
                    if (platform.state === 'stable') {
                        platform.state = 'shaking';
                        platform.fallTimer = 0;
                    }
                }
            }
            // Bouncy platforms: give extra jump
            else if (platform.type === 'bouncy') {
                if (player.velocityY > 0 &&
                    player.y + player.height - player.velocityY * deltaTime <= platform.y + 5) {
                    player.y = platform.y - player.height;
                    player.velocityY = platform.bounceForce; // Super jump!
                    player.onGround = false; // Immediately in air
                    platform.squish = 0.5; // Visual squish effect

                    // Particle effect
                    for (let i = 0; i < 5; i++) {
                        particles.push({
                            x: player.x + player.width / 2,
                            y: player.y + player.height,
                            velocityY: 50 + Math.random() * 100,
                            lifetime: 0.5,
                            elapsed: 0,
                            type: 'star'
                        });
                    }
                }
            }
            // Rainbow platforms: bouncy + bonus points
            else if (platform.type === 'rainbow') {
                if (player.velocityY > 0 &&
                    player.y + player.height - player.velocityY * deltaTime <= platform.y + 5) {
                    player.y = platform.y - player.height;
                    player.velocityY = platform.bounceForce; // Bounce!
                    player.onGround = false;
                    stats.bonusScore += 5; // Bonus points!

                    // Rainbow particles
                    for (let i = 0; i < 6; i++) {
                        particles.push({
                            x: player.x + player.width / 2,
                            y: player.y + player.height,
                            velocityY: 30 + Math.random() * 80,
                            lifetime: 0.6,
                            elapsed: 0,
                            type: 'rainbow'
                        });
                    }
                }
            }
            // Solid platforms: land from above only (same as before)
            else if (platform.type === 'solid') {
                if (player.velocityY > 0 &&
                    player.y + player.height - player.velocityY * deltaTime <= platform.y + 10) {
                    player.y = platform.y - player.height;
                    player.velocityY = 0;
                    player.onGround = true;
                    player.currentPlatform = platform;
                }
            }
            // Island platforms: floating islands you can land on from above
            else if (platform.type === 'island') {
                if (player.velocityY > 0 &&
                    player.y + player.height - player.velocityY * deltaTime <= platform.y + 10) {
                    player.y = platform.y - player.height;
                    player.velocityY = 0;
                    player.onGround = true;
                    player.currentPlatform = platform;
                }
            }
        }
    }

    // Moving platform collision (Phase 5)
    for (const platform of movingPlatforms) {
        if (checkCollision(player, platform)) {
            // Only land if coming from above
            if (player.velocityY > 0 &&
                player.y + player.height - player.velocityY * deltaTime <= platform.y + 5) {
                player.y = platform.y - player.height;
                player.velocityY = 0;
                player.onGround = true;
                player.currentPlatform = platform;

                // Transfer platform velocity to player
                player.x += platform.velocityX * deltaTime;
                player.y += platform.velocityY * deltaTime;
            }
        }
    }

    // Slope collision (Phase 5)
    for (const slope of slopes) {
        // Check if player is horizontally on the slope
        if (player.x + player.width > slope.x &&
            player.x < slope.x + slope.width) {

            // Calculate player's position on the slope (0 to 1)
            const playerCenterX = player.x + player.width / 2;
            const slopeProgress = (playerCenterX - slope.x) / slope.width;

            // Calculate Y position on slope based on X position
            const slopeY = slope.y + (slope.endY - slope.y) * slopeProgress;

            // Improved collision: prevent clipping through slope
            // Check if player is intersecting with the slope surface
            if (player.y + player.height > slopeY - 5 && player.y < slopeY + 30) {
                // If player is falling or on the slope
                if (player.velocityY >= 0 || player.y + player.height >= slopeY - 2) {
                    player.y = slopeY - player.height;
                    player.velocityY = 0;
                    player.onGround = true;
                    player.currentPlatform = slope;

                    // Slide down steep slopes (angle > 25 degrees)
                    if (Math.abs(slope.angle) > 25 && Math.abs(player.velocityX) < 50) {
                        const slideDirection = slope.direction === 'up' ? -1 : 1;
                        player.x += slideDirection * 80 * deltaTime;
                    }
                }
            }
        }
    }

    // Clear current platform if not on ground
    if (!player.onGround) {
        player.currentPlatform = null;
    }

    // Obstacle collision (fire hydrants - only if solid)
    for (const obstacle of obstacles) {
        if (obstacle.solid !== false && checkCollision(player, obstacle)) {
            // Block horizontal movement
            if (player.velocityX > 0) {
                player.x = obstacle.x - player.width;
            } else if (player.velocityX < 0) {
                player.x = obstacle.x + obstacle.width;
            }
            player.velocityX = 0;
        }
    }

    // Checkpoint collision (dog beds)
    for (let i = 0; i < checkpoints.length; i++) {
        const checkpoint = checkpoints[i];
        if (!checkpoint.activated && checkCollision(player, checkpoint)) {
            checkpoint.activated = true;
            currentCheckpoint = i;
            console.log(`🛏️ Checkpoint ${i + 1} activated!`);
        }
    }

    // Collectible collision (with point variations and stats)
    for (const collectible of collectibles) {
        if (!collectible.collected && checkCollision(player, collectible)) {
            collectible.collected = true;

            // Points
            if (collectible.points) {
                score += collectible.points;
                document.getElementById('score').textContent = `Score: ${score}`;
            }

            // Happiness (Phase 4)
            if (collectible.happinessGain) {
                stats.happiness = Math.min(stats.maxHappiness, stats.happiness + collectible.happinessGain);
            }

            // Energy (Phase 4)
            if (collectible.energyGain) {
                stats.energy = Math.min(stats.maxEnergy, stats.energy + collectible.energyGain);
            }

            // XP (Phase 4)
            if (collectible.xpGain) {
                gainXP(collectible.xpGain);
            }

            // Energy restoration when tired (all collectibles restore some energy!)
            let energyGain = 0;
            const isTired = stats.energy < 30;

            // Collection tracking (Phase 4)
            if (collectible.type === 'bone') {
                collectionStats.bonesCollected++;
                energyGain = isTired ? 15 : 8;
                console.log(`🦴 Bone collected! ${collectionStats.bonesCollected}/${collectionStats.bonesTotal} +${energyGain} energy`);
            } else if (collectible.type === 'ball') {
                collectionStats.ballsCollected++;
                energyGain = isTired ? 20 : 10;
                console.log(`⚾ Ball collected! +20 points, +15 happiness, +${energyGain} energy`);
            } else if (collectible.type === 'frisbee') {
                collectionStats.frisbeesCollected++;
                energyGain = isTired ? 25 : 12;
                console.log(`🥏 Frisbee collected! +30 points, +20 happiness, +${energyGain} energy`);
            } else if (collectible.type === 'water') {
                collectionStats.waterBowlsCollected++;
                energyGain = 30;
                console.log('💧 Water bowl! +30 energy');
            } else if (collectible.type === 'food') {
                collectionStats.foodBowlsCollected++;
                energyGain = 50;
                console.log('🍖 Food bowl! +50 energy');
            } else if (collectible.type === 'apple') {
                stats.happiness = Math.min(stats.maxHappiness, stats.happiness + 15);
                energyGain = isTired ? 18 : 10;
                console.log(`🍎 Apple collected! +15 points, +15 happiness, +${energyGain} energy`);
            } else if (collectible.type === 'lightning') {
                stats.bonusScore += 50;
                energyGain = isTired ? 20 : 8;

                // Activate speed powerup!
                powerup.active = true;
                powerup.type = 'speed';
                powerup.timeLeft = powerup.duration;

                // Switch to dash animation (frames 17-20)
                useFrameSet2 = true;

                // Lightning bolts respawn after 15 seconds
                collectible.respawnTimer = 15;
                collectible.respawnDuration = 15;

                console.log(`⚡ LIGHTNING POWERUP! +50 bonus, +${energyGain} energy, 2X SPEED for ${powerup.duration}s! DASH MODE ACTIVATED!`);
            }

            // Apply energy gain
            if (energyGain > 0) {
                stats.energy = Math.min(stats.maxEnergy, stats.energy + energyGain);
            }

            // Check achievements
            checkAchievements();
        }
    }

    // Update collectible animations
    for (const collectible of collectibles) {
        if (!collectible.collected) {
            collectible.floatTime += deltaTime * 2;

            // Type-specific animations
            if (collectible.type === 'bone') {
                // Standard float
                collectible.y = collectible.initialY + Math.sin(collectible.floatTime) * 10;
            } else if (collectible.type === 'ball') {
                // Bouncing animation
                collectible.bounceTime += deltaTime * 3;
                collectible.y = collectible.initialY + Math.abs(Math.sin(collectible.bounceTime)) * 20;
            } else if (collectible.type === 'frisbee') {
                // Arc flight animation
                collectible.arcTime += deltaTime;
                collectible.y = collectible.initialY + Math.sin(collectible.arcTime) * collectible.arcHeight;
            } else if (collectible.type === 'apple') {
                // Gentle float like bones
                collectible.y = collectible.initialY + Math.sin(collectible.floatTime) * 8;
            } else if (collectible.type === 'lightning') {
                // Electric pulse animation
                collectible.pulseTime = (collectible.pulseTime || 0) + deltaTime * 5;
                collectible.y = collectible.initialY + Math.sin(collectible.pulseTime) * 15;
            }
        } else if (collectible.respawnTimer !== undefined) {
            // Handle respawning collectibles (lightning bolts)
            collectible.respawnTimer -= deltaTime;

            if (collectible.respawnTimer <= 0) {
                // Respawn the collectible!
                collectible.collected = false;
                collectible.respawnTimer = undefined;
                console.log('⚡ Lightning bolt respawned!');

                // Spawn respawn particles
                for (let i = 0; i < 8; i++) {
                    particles.push({
                        x: collectible.x + collectible.width / 2,
                        y: collectible.y + collectible.height / 2,
                        velocityX: Math.cos(i * Math.PI / 4) * 100,
                        velocityY: Math.sin(i * Math.PI / 4) * 100,
                        lifetime: 0.8,
                        elapsed: 0,
                        type: 'lightning'
                    });
                }
            }
        }
    }

    // Fall detection (respawn at checkpoint)
    if (player.y > CANVAS_HEIGHT + 100) {
        const checkpoint = checkpoints[currentCheckpoint];
        player.x = checkpoint.x;
        player.y = checkpoint.y - player.height;
        player.velocityX = 0;
        player.velocityY = 0;
        console.log('💀 Fell off! Respawning at checkpoint...');
    }

    // NPC interactions (Phase 3)
    updateNPCs(deltaTime);

    // Squirrel chase (Phase 3)
    updateSquirrel(deltaTime);

    // Update particles (hearts)
    updateParticles(deltaTime);

    // Update moving platforms (Phase 5)
    updateMovingPlatforms(deltaTime);

    // Update ladder climbing (Phase 5)
    updateLadders(deltaTime);

    // Update falling and bouncy platforms (Phase 5)
    updateSpecialPlatforms(deltaTime);

    // Update zones and interactive elements (Phase 5)
    updateZones();
    updateInteractiveElements(deltaTime);
    updateWaterZones(deltaTime);
    updateWallJump(deltaTime);

    // Update stats (Phase 4)
    updateStats(deltaTime);

    // Update animation
    updateAnimation(deltaTime);

    // Update camera (follow player)
    camera.x = player.x - CANVAS_WIDTH / 2;

    // Sky area: Camera follows player vertically (parallax to sky platforms!)
    if (player.x > 3400) {
        // Vertical camera follow in sky zone with smooth interpolation
        const targetY = player.y - CANVAS_HEIGHT / 2;
        const lerpSpeed = 0.08; // Smooth camera movement
        camera.y += (targetY - camera.y) * lerpSpeed;

        // Clamp vertical camera (don't go too low or too high)
        camera.y = Math.max(-200, Math.min(camera.y, GROUND_Y - 100));
    } else {
        // Normal horizontal platformer - smoothly return to Y = 0
        const lerpSpeed = 0.1;
        camera.y += (0 - camera.y) * lerpSpeed;
    }

    // Clamp horizontal camera to level bounds
    camera.x = Math.max(0, Math.min(camera.x, CANVAS_WIDTH * 4 - CANVAS_WIDTH));
}

// ===== NPC UPDATE (Phase 3) =====
function updateNPCs(deltaTime) {
    const INTERACT_RANGE = 80; // Pixels for greeting

    for (const npc of npcs) {
        // Patrol AI
        if (npc.type === 'patrol') {
            npc.x += npc.speed * npc.direction * deltaTime;

            // Boundary check
            if (npc.x >= npc.maxX) {
                npc.direction = -1;
                npc.facingRight = false;
            }
            if (npc.x <= npc.minX) {
                npc.direction = 1;
                npc.facingRight = true;
            }
        }

        // Flying patrol AI (for sky birds)
        if (npc.type === 'flying-patrol') {
            // Figure-8 or circular flight pattern
            if (!npc.flightTime) npc.flightTime = 0;
            npc.flightTime += deltaTime;

            // Horizontal movement
            npc.x += npc.speed * npc.direction * deltaTime;
            if (npc.x >= npc.maxX) npc.direction = -1;
            if (npc.x <= npc.minX) npc.direction = 1;
            npc.facingRight = npc.direction > 0;

            // Vertical bobbing
            const centerY = (npc.minY + npc.maxY) / 2;
            const amplitude = (npc.maxY - npc.minY) / 2;
            npc.y = centerY + Math.sin(npc.flightTime * 2) * amplitude;
        }

        // Proximity interaction (greeting)
        if (!npc.greeted) {
            const distance = Math.abs(player.x - npc.x);

            if (distance < INTERACT_RANGE) {
                npc.greeted = true;
                score += 15; // Bonus for greeting
                document.getElementById('score').textContent = `Score: ${score}`;

                // Phase 4: Stats
                stats.happiness = Math.min(stats.maxHappiness, stats.happiness + 15);
                gainXP(15);
                collectionStats.npcsGreeted++;

                console.log(`🐶 Met ${npc.name}! +15 points, +15 happiness`);

                // Spawn heart particle
                particles.push({
                    x: npc.x + npc.width / 2,
                    y: npc.y - 20,
                    velocityY: -50,
                    lifetime: 1.5,
                    elapsed: 0,
                    type: 'heart'
                });

                // Trigger bark animation
                npc.barkTime = 0;

                // Check achievements
                checkAchievements();
            }
        }

        // Bark animation (scale pulse)
        if (npc.greeted && npc.barkTime < 1) {
            npc.barkTime += deltaTime * 3;
        }
    }
}

// ===== SQUIRREL UPDATE (Phase 3) =====
function updateSquirrel(deltaTime) {
    if (!squirrel || squirrel.caught) {
        // Respawn timer
        if (squirrel && squirrel.caught) {
            squirrel.respawnTimer += deltaTime;
            if (squirrel.respawnTimer > 30) {
                // Respawn after 30 seconds
                squirrel.caught = false;
                squirrel.respawnTimer = 0;
                squirrel.x = 750;
                squirrel.y = platforms[5].y - 180;
                squirrel.state = 'idle';
                squirrel.chasePhase = 0;
                console.log('🐿️ Squirrel returned to tree!');
            }
        }
        return;
    }

    const distanceToPlayer = Math.abs(player.x - squirrel.x);

    // State machine
    if (squirrel.state === 'idle') {
        // Alert when player gets close
        if (distanceToPlayer < squirrel.alertDistance) {
            squirrel.state = 'alerted';
            squirrel.targetX = 900; // Jump to ground and run
            squirrel.y = GROUND_Y - 25;
            squirrel.chasePhase = 1;
            console.log('🐿️ Squirrel noticed you!');
        }
    } else if (squirrel.state === 'alerted') {
        // Run to target
        const dir = squirrel.targetX > squirrel.x ? 1 : -1;
        squirrel.x += squirrel.speed * dir * deltaTime;

        // Reached target
        if (Math.abs(squirrel.x - squirrel.targetX) < 10) {
            squirrel.state = 'running';
        }

        // Player catches during run
        if (distanceToPlayer < 30) {
            squirrel.caught = true;
            score += 50;
            document.getElementById('score').textContent = `Score: ${score}`;

            // Phase 4: Stats
            stats.happiness = Math.min(stats.maxHappiness, stats.happiness + 25);
            gainXP(50);
            collectionStats.squirrelCaught = true;

            console.log('🐿️ Caught the squirrel! +50 points, +25 happiness!');

            // Spawn celebration particles
            for (let i = 0; i < 3; i++) {
                particles.push({
                    x: squirrel.x,
                    y: squirrel.y,
                    velocityY: -60 - i * 20,
                    lifetime: 2,
                    elapsed: 0,
                    type: 'star'
                });
            }

            // Check achievements
            checkAchievements();
        }
    } else if (squirrel.state === 'running') {
        // Player approaches again
        if (distanceToPlayer < 100) {
            if (squirrel.chasePhase === 1) {
                squirrel.targetX = 1050;
                squirrel.state = 'alerted';
                squirrel.chasePhase = 2;
            } else if (squirrel.chasePhase === 2) {
                squirrel.targetX = 1200;
                squirrel.state = 'alerted';
                squirrel.chasePhase = 3;
            }
        }

        // Player catches
        if (distanceToPlayer < 30) {
            squirrel.caught = true;
            score += 50;
            document.getElementById('score').textContent = `Score: ${score}`;
            console.log('🐿️ Caught the squirrel! +50 points!');
        }
    }
}

// ===== PARTICLE UPDATE (Phase 3) =====
function updateParticles(deltaTime) {
    for (let i = particles.length - 1; i >= 0; i--) {
        const p = particles[i];
        p.elapsed += deltaTime;
        p.y += p.velocityY * deltaTime;

        // Remove expired particles
        if (p.elapsed >= p.lifetime) {
            particles.splice(i, 1);
        }
    }
}

// ===== MOVING PLATFORMS UPDATE (Phase 5) =====
function updateMovingPlatforms(deltaTime) {
    for (const platform of movingPlatforms) {
        // Reset velocity for this frame
        platform.velocityX = 0;
        platform.velocityY = 0;

        if (platform.type === 'horizontal') {
            // Move left-right
            const oldX = platform.x;
            platform.x += platform.speed * platform.direction * deltaTime;

            // Calculate actual velocity (for player transfer)
            platform.velocityX = (platform.x - oldX) / deltaTime;

            // Bounce at boundaries
            if (platform.x >= platform.maxX) {
                platform.x = platform.maxX;
                platform.direction = -1;
            }
            if (platform.x <= platform.minX) {
                platform.x = platform.minX;
                platform.direction = 1;
            }
        }
        else if (platform.type === 'vertical') {
            // Move up-down
            const oldY = platform.y;
            platform.y += platform.speed * platform.direction * deltaTime;

            // Calculate actual velocity (for player transfer)
            platform.velocityY = (platform.y - oldY) / deltaTime;

            // Bounce at boundaries
            if (platform.y >= platform.maxY) {
                platform.y = platform.maxY;
                platform.direction = -1;
            }
            if (platform.y <= platform.minY) {
                platform.y = platform.minY;
                platform.direction = 1;
            }
        }
        else if (platform.type === 'circular') {
            // Orbit around center point
            const oldX = platform.x;
            const oldY = platform.y;

            platform.angle += platform.angularSpeed * deltaTime;
            platform.x = platform.centerX + Math.cos(platform.angle) * platform.radius - platform.width / 2;
            platform.y = platform.centerY + Math.sin(platform.angle) * platform.radius - platform.height / 2;

            // Calculate velocity
            platform.velocityX = (platform.x - oldX) / deltaTime;
            platform.velocityY = (platform.y - oldY) / deltaTime;
        }
    }
}

// ===== LADDER CLIMBING UPDATE (Phase 5) =====
function updateLadders(deltaTime) {
    const CLIMB_SPEED = 150;

    // Check if player is touching a ladder
    let touchingLadder = null;
    for (const ladder of ladders) {
        if (checkCollision(player, ladder)) {
            touchingLadder = ladder;
            break;
        }
    }

    // Ladder controls
    if (touchingLadder) {
        // Press UP to start climbing
        if (keys.up && !player.climbing) {
            player.climbing = true;
            player.currentLadder = touchingLadder;
            player.velocityY = 0;
            player.velocityX = 0;
            // Center player on ladder
            player.x = touchingLadder.x + touchingLadder.width / 2 - player.width / 2;
        }

        // If already climbing, handle movement
        if (player.climbing && player.currentLadder === touchingLadder) {
            // Disable gravity while climbing
            player.velocityY = 0;

            // Climb up/down
            if (keys.up) {
                player.y -= CLIMB_SPEED * deltaTime;
            }
            if (keys.down) {
                player.y += CLIMB_SPEED * deltaTime;
            }

            // Jump off ladder
            if (keys.jump) {
                player.climbing = false;
                player.currentLadder = null;
                player.velocityY = PLAYER.jumpForce * 0.7; // Smaller jump from ladder
            }

            // Dismount at top
            if (player.y < touchingLadder.y) {
                player.climbing = false;
                player.currentLadder = null;
            }

            // Dismount at bottom if on ground
            if (player.onGround) {
                player.climbing = false;
                player.currentLadder = null;
            }
        }
    } else {
        // Not touching any ladder
        if (player.climbing) {
            player.climbing = false;
            player.currentLadder = null;
        }
    }
}

// ===== SPECIAL PLATFORMS UPDATE (Phase 5) =====
function updateSpecialPlatforms(deltaTime) {
    for (const platform of platforms) {
        // Update falling platforms
        if (platform.type === 'falling') {
            platform.fallTimer += deltaTime;

            if (platform.state === 'shaking') {
                // Shake for fallDelay seconds, then fall
                if (platform.fallTimer >= platform.fallDelay) {
                    platform.state = 'falling';
                    platform.fallTimer = 0;
                    platform.fallVelocity = 0;
                }
            }
            else if (platform.state === 'falling') {
                // Fall down with acceleration
                platform.fallVelocity += 800 * deltaTime;
                platform.y += platform.fallVelocity * deltaTime;

                // After falling off screen, start respawn timer
                if (platform.y > CANVAS_HEIGHT + 100) {
                    platform.state = 'respawning';
                    platform.fallTimer = 0;
                }
            }
            else if (platform.state === 'respawning') {
                // Wait respawnTime seconds, then respawn
                if (platform.fallTimer >= platform.respawnTime) {
                    platform.state = 'stable';
                    platform.y = platform.originalY;
                    platform.fallTimer = 0;
                    platform.fallVelocity = 0;
                }
            }
        }

        // Update bouncy platforms (squish effect)
        if (platform.type === 'bouncy' && platform.squish > 0) {
            platform.squish = Math.max(0, platform.squish - deltaTime * 2);
        }
    }
}

// ===== ZONES UPDATE (Phase 5) =====
function updateZones() {
    // Determine which zone player is in based on X position
    for (let i = 0; i < zones.length; i++) {
        const zone = zones[i];
        if (player.x >= zone.startX && player.x < zone.endX) {
            if (currentZone !== i) {
                currentZone = i;
                console.log(`🌍 Entered Zone ${i + 1}: ${zone.name}`);
            }
            break;
        }
    }
}

// ===== INTERACTIVE ELEMENTS UPDATE (Phase 5) =====
function updateInteractiveElements(deltaTime) {
    for (const element of interactiveElements) {
        if (element.type === 'switch' || element.type === 'lever') {
            // Check if player is touching the switch/lever
            if (checkCollision(player, element)) {
                // Activate with UP key
                if (keys.up && !element.activated) {
                    element.activated = true;
                    console.log(`🔘 ${element.type === 'switch' ? 'Switch' : 'Lever'} activated!`);

                    // Trigger linked elements
                    for (const linked of interactiveElements) {
                        if (linked.type === 'door' && !linked.requiresKey) {
                            linked.open = true;
                        }
                        if (linked.type === 'bridge') {
                            linked.extended = true;
                        }
                    }
                }
            }
        }

        // Door collision (solid when closed)
        if (element.type === 'door' && !element.open) {
            if (checkCollision(player, element)) {
                // Block player movement
                if (player.velocityX > 0) {
                    player.x = element.x - player.width;
                } else if (player.velocityX < 0) {
                    player.x = element.x + element.width;
                }
                player.velocityX = 0;
            }
        }

        // Bridge collision (acts as platform when extended)
        if (element.type === 'bridge' && element.extended) {
            if (player.velocityY > 0 &&
                player.x + player.width > element.x &&
                player.x < element.x + element.width &&
                player.y + player.height >= element.y &&
                player.y + player.height - player.velocityY * deltaTime <= element.y + 5) {
                player.y = element.y - player.height;
                player.velocityY = 0;
                player.onGround = true;
            }
        }
    }
}

// ===== WATER ZONES UPDATE (Phase 5) =====
function updateWaterZones(deltaTime) {
    let inWater = false;
    let currentWaterZone = null;

    // Check if player is in any water zone
    for (const water of waterZones) {
        if (player.x + player.width > water.x &&
            player.x < water.x + water.width &&
            player.y + player.height > water.y &&
            player.y < water.y + water.height) {
            inWater = true;
            currentWaterZone = water;
            break;
        }
    }

    if (inWater && currentWaterZone) {
        // Swimming mechanics
        // Reduce gravity, apply buoyancy
        player.velocityY -= currentWaterZone.buoyancy * deltaTime;

        // Apply drag to movement
        player.velocityX *= currentWaterZone.drag;
        player.velocityY *= 0.95; // Slow vertical movement in water

        // Allow jumping out of water with stronger jump
        if ((keys.jump || keys.up) && !player.hasJumped) {
            player.velocityY = -400; // Strong jump to exit water
            player.hasJumped = true;
        }

        if (!keys.jump && !keys.up) {
            player.hasJumped = false;
        }

        // Prevent sinking below water bottom
        if (player.y + player.height > currentWaterZone.y + currentWaterZone.height) {
            player.y = currentWaterZone.y + currentWaterZone.height - player.height;
            player.velocityY = Math.min(player.velocityY, 0);
        }

        // Show swimming indicator
        if (Math.random() < 0.1) {
            particles.push({
                x: player.x + player.width / 2 + (Math.random() - 0.5) * 40,
                y: player.y + player.height - 10,
                velocityY: -30,
                lifetime: 0.8,
                elapsed: 0,
                type: 'water' // Water bubble
            });
        }
    }
}

// ===== WALL JUMP UPDATE (Phase 5) =====
function updateWallJump(deltaTime) {
    // Check if player is touching a wall while in air
    if (!player.onGround && !player.climbing) {
        let touchingWall = false;
        let wallDirection = 0; // -1 = left wall, 1 = right wall

        // Check platform walls
        for (const platform of platforms) {
            // Skip non-solid platforms
            if (platform.type !== 'solid') continue;

            // Check left side of platform
            if (player.x + player.width >= platform.x - 5 &&
                player.x + player.width <= platform.x + 10 &&
                player.y + player.height > platform.y &&
                player.y < platform.y + platform.height) {
                touchingWall = true;
                wallDirection = -1; // Wall is on player's right
            }

            // Check right side of platform
            if (player.x <= platform.x + platform.width + 5 &&
                player.x >= platform.x + platform.width - 10 &&
                player.y + player.height > platform.y &&
                player.y < platform.y + platform.height) {
                touchingWall = true;
                wallDirection = 1; // Wall is on player's left
            }
        }

        // Check obstacle walls (fire hydrants - only if solid)
        for (const obstacle of obstacles) {
            if (obstacle.solid !== false) {
                if (player.x + player.width >= obstacle.x - 5 &&
                    player.x + player.width <= obstacle.x + 10 &&
                    player.y + player.height > obstacle.y &&
                    player.y < obstacle.y + obstacle.height) {
                    touchingWall = true;
                    wallDirection = -1;
                }

                if (player.x <= obstacle.x + obstacle.width + 5 &&
                    player.x >= obstacle.x + obstacle.width - 10 &&
                    player.y + player.height > obstacle.y &&
                    player.y < obstacle.y + obstacle.height) {
                    touchingWall = true;
                    wallDirection = 1;
                }
            }
        }

        if (touchingWall) {
            // Wall slide (slow fall)
            if (player.velocityY > 50) {
                player.velocityY = 50; // Max slide speed
            }

            // Wall jump (press jump while touching wall)
            if (keys.jump && keys.jumpPressed) {
                player.velocityY = PLAYER.jumpForce * 0.9; // Slightly weaker jump
                player.velocityX = wallDirection * 250; // Push away from wall
                player.facingRight = wallDirection > 0;

                console.log('🧗 Wall jump!');

                // Particle effect
                for (let i = 0; i < 3; i++) {
                    particles.push({
                        x: player.x + player.width / 2,
                        y: player.y + player.height / 2,
                        velocityY: -50 - i * 30,
                        lifetime: 0.5,
                        elapsed: 0,
                        type: 'star'
                    });
                }
            }
        }
    }
}

// ===== STATS UPDATE (Phase 4) =====
function updateStats(deltaTime) {
    // Energy decay (constant drain)
    stats.energy = Math.max(0, stats.energy - 3 * deltaTime);

    // Happiness decay when idle
    if (Math.abs(player.velocityX) < 10 && player.onGround) {
        stats.idleTime += deltaTime;
        if (stats.idleTime > 1) {
            stats.happiness = Math.max(0, stats.happiness - 5 * deltaTime);
        }
    } else {
        stats.idleTime = 0;
        // Slight happiness gain when moving
        stats.happiness = Math.min(stats.maxHappiness, stats.happiness + 1 * deltaTime);
    }

    // Update achievement popup timer
    if (achievementPopup) {
        achievementPopup.timeLeft -= deltaTime;
        if (achievementPopup.timeLeft <= 0) {
            achievementPopup = null;
        }
    }

    // Update powerup timer
    if (powerup.active) {
        powerup.timeLeft -= deltaTime;
        if (powerup.timeLeft <= 0) {
            powerup.active = false;
            powerup.type = null;
            powerup.timeLeft = 0;

            // Restore normal animation (frames 1-16)
            useFrameSet2 = false;

            console.log('⚡ Speed powerup expired! Normal speed restored.');
        }
    }

    // Clamp stats
    stats.happiness = Math.max(0, Math.min(stats.maxHappiness, stats.happiness));
    stats.energy = Math.max(0, Math.min(stats.maxEnergy, stats.energy));
}

function gainXP(amount) {
    stats.xp += amount;

    // Level up thresholds
    const levelThresholds = [0, 100, 250, 450, 700];
    const newLevel = levelThresholds.findIndex((threshold, i) => {
        return stats.xp >= threshold && stats.xp < (levelThresholds[i + 1] || Infinity);
    }) + 1;

    if (newLevel > stats.level) {
        stats.level = newLevel;
        console.log(`🎉 Level Up! Now Level ${stats.level}`);

        // Level bonuses
        if (stats.level === 2) stats.maxHappiness += 10;
        if (stats.level === 3) stats.maxEnergy += 10;
        if (stats.level === 4) console.log('Speed +5% unlocked!');
        if (stats.level === 5) console.log('🏆 MAX LEVEL! All stats boosted!');
    }
}

function checkAchievements() {
    // First Friend
    if (!achievementDefs[0].unlocked && collectionStats.npcsGreeted >= 1) {
        unlockAchievement(0);
    }

    // Social Butterfly
    if (!achievementDefs[1].unlocked && collectionStats.npcsGreeted >= 6) {
        unlockAchievement(1);
    }

    // Bone Collector
    if (!achievementDefs[2].unlocked && collectionStats.bonesCollected >= 12) {
        unlockAchievement(2);
    }

    // Toy Master
    if (!achievementDefs[3].unlocked &&
        collectionStats.ballsCollected >= 4 &&
        collectionStats.frisbeesCollected >= 2) {
        unlockAchievement(3);
    }

    // Squirrel Whisperer
    if (!achievementDefs[4].unlocked && collectionStats.squirrelCaught) {
        unlockAchievement(4);
    }

    // Perfect Score
    if (!achievementDefs[5].unlocked && score >= 400) {
        unlockAchievement(5);
    }
}

function unlockAchievement(index) {
    const achievement = achievementDefs[index];
    if (achievement.unlocked) return;

    achievement.unlocked = true;
    achievements.push(achievement.id);

    // Show popup
    achievementPopup = {
        achievement: achievement,
        timeLeft: 3 // 3 seconds
    };

    // Bonus rewards
    stats.happiness = Math.min(stats.maxHappiness, stats.happiness + 10);
    gainXP(100);

    console.log(`🏆 ACHIEVEMENT UNLOCKED: ${achievement.icon} ${achievement.name}!`);
    console.log(`   ${achievement.desc}`);
}

// ===== COLLISION DETECTION =====
function checkCollision(a, b) {
    return a.x < b.x + b.width &&
        a.x + a.width > b.x &&
        a.y < b.y + b.height &&
        a.y + a.height > b.y;
}

// ===== ANIMATION UPDATE =====
function updateAnimation(deltaTime) {
    // Determine animation state
    if (!player.onGround && player.velocityY < 0) {
        player.animation = 'jump';
    } else if (!player.onGround && player.velocityY > 0) {
        player.animation = 'fall';
    } else if (Math.abs(player.velocityX) > 10) {
        player.animation = 'run';
    } else {
        player.animation = 'idle';
    }

    // Get animation frames (based on 4x4 sprite sheet grid)
    // FRAME SET 1 (Original: dog_1 to dog_16)
    // Idle: dog_5
    // Running loop: dog_6 → dog_7 → dog_4 → dog_3 (repeats without dog_5)
    // Jump/Fall: Same as running (legs keep moving in air)
    // Frame indices (0-indexed): 5, 6, 3, 2
    const animationsSet1 = {
        idle: [4],                 // dog_5 - standing frame (idle only)
        run: [5, 6, 3, 2],         // dog_6 → dog_7 → dog_4 → dog_3 (continuous loop)
        jump: [5, 6, 3, 2],        // Same running cycle while jumping
        fall: [5, 6, 3, 2]         // Same running cycle while falling
    };

    // FRAME SET 2 (New: dog_17 to dog_20)
    // Using frames 17-20 for alternate animation
    // Frame indices (0-indexed): 16, 17, 18, 19
    const animationsSet2 = {
        idle: [16],                // dog_17 - standing frame
        run: [17, 18, 19, 16],     // dog_18 → dog_19 → dog_20 → dog_17 (continuous loop)
        jump: [17, 18, 19, 16],    // Same running cycle while jumping
        fall: [17, 18, 19, 16]     // Same running cycle while falling
    };

    // Select active animation set based on toggle state
    const activeAnimations = useFrameSet2 ? animationsSet2 : animationsSet1;
    const frames = activeAnimations[player.animation];

    // Update frame timing
    if (frames.length > 1) {
        player.animationTime += deltaTime * 1000;

        if (player.animationTime >= FRAME_TIME) {
            player.currentFrame = (player.currentFrame + 1) % frames.length;
            player.animationTime = 0;
        }
    } else {
        player.currentFrame = 0;
    }

    player.spriteIndex = frames[player.currentFrame];
}

// ===== RENDER =====
function render() {
    // Clear canvas
    ctx.clearRect(0, 0, CANVAS_WIDTH, CANVAS_HEIGHT);

    // Draw sky background (zone-based color)
    const zone = zones[currentZone] || zones[0];
    const gradient = ctx.createLinearGradient(0, 0, 0, CANVAS_HEIGHT);
    gradient.addColorStop(0, zone.skyGradientTop);
    gradient.addColorStop(1, zone.skyGradientBottom);
    ctx.fillStyle = gradient;
    ctx.fillRect(0, 0, CANVAS_WIDTH, CANVAS_HEIGHT);

    // Zone ambient overlay (subtle tint only)
    // Disabled - was too intrusive
    // ctx.fillStyle = zone.ambientColor;
    // ctx.fillRect(0, 0, CANVAS_WIDTH, CANVAS_HEIGHT);

    // Draw decorative sky swirls (like in reference images)
    ctx.save();
    ctx.globalAlpha = 0.15;
    ctx.strokeStyle = 'rgba(255, 255, 255, 0.3)';
    ctx.lineWidth = 3;

    // Create swirl patterns across the sky
    const swirlCount = 8;
    for (let i = 0; i < swirlCount; i++) {
        const swirlX = (i * 300 + (camera.x * 0.1)) % (CANVAS_WIDTH + 200) - 100;
        const swirlY = 80 + (i % 3) * 100;
        const radius = 40 + (i % 2) * 20;

        ctx.beginPath();
        ctx.arc(swirlX, swirlY, radius, 0, Math.PI * 1.5);
        ctx.stroke();

        // Inner swirl
        ctx.beginPath();
        ctx.arc(swirlX - 5, swirlY - 5, radius * 0.6, Math.PI * 0.5, Math.PI * 2);
        ctx.stroke();
    }
    ctx.restore();

    // Draw background parallax layers (Phase 5)
    for (const layer of backgroundLayers) {
        if (layer.type === 'mountains') {
            ctx.fillStyle = zone.mountainColor || layer.color; // Use zone mountain color
            for (const mountain of layer.mountains) {
                const parallaxX = mountain.x - (camera.x * layer.parallaxSpeed);

                // Draw mountain as triangle
                ctx.beginPath();
                ctx.moveTo(parallaxX, layer.yOffset); // Peak
                ctx.lineTo(parallaxX - mountain.width / 2, layer.yOffset + mountain.height); // Bottom left
                ctx.lineTo(parallaxX + mountain.width / 2, layer.yOffset + mountain.height); // Bottom right
                ctx.closePath();
                ctx.fill();

                // Add subtle gradient for depth
                ctx.fillStyle = 'rgba(255, 255, 255, 0.1)';
                ctx.beginPath();
                ctx.moveTo(parallaxX, layer.yOffset);
                ctx.lineTo(parallaxX - mountain.width / 4, layer.yOffset + mountain.height / 2);
                ctx.lineTo(parallaxX, layer.yOffset + mountain.height);
                ctx.closePath();
                ctx.fill();

                ctx.fillStyle = layer.color; // Reset color
            }
        }
        else if (layer.type === 'trees') {
            // Draw distant small trees
            for (const treeX of layer.positions) {
                const parallaxX = treeX - (camera.x * layer.parallaxSpeed);
                const treeSprite = treeSprites[1]; // Small tree

                if (treeSprite && treeSprite.complete) {
                    ctx.globalAlpha = 0.4; // Faded for distance
                    ctx.drawImage(
                        treeSprite,
                        parallaxX,
                        layer.yOffset,
                        40,
                        60
                    );
                    ctx.globalAlpha = 1.0;
                }
            }
        }
    }

    // Draw sun (top-right corner, no parallax)
    if (sunSprite && sunSprite.complete) {
        ctx.drawImage(
            sunSprite,
            CANVAS_WIDTH - 150, // 150px from right edge
            50,                  // 50px from top
            100,                 // width
            100                  // height
        );
    }

    // Draw clouds (background layer with parallax)
    for (const cloud of clouds) {
        const cloudSprite = cloudSprites[cloud.variant];
        if (cloudSprite && cloudSprite.complete) {
            // Parallax effect: clouds move slower than camera
            const parallaxX = cloud.x - (camera.x * cloud.speed);
            ctx.drawImage(
                cloudSprite,
                parallaxX,
                cloud.y,
                cloud.width,
                cloud.height
            );
        }
    }

    // Draw platforms
    for (const platform of platforms) {
        ctx.save();

        // Special visual effects for platform types
        let offsetX = 0, offsetY = 0;

        // Falling platform shake effect
        if (platform.type === 'falling' && platform.state === 'shaking') {
            offsetX = (Math.random() - 0.5) * 6;
            offsetY = (Math.random() - 0.5) * 6;
        }

        // Bouncy platform squish effect
        if (platform.type === 'bouncy' && platform.squish > 0) {
            const squishAmount = platform.squish;
            ctx.translate(
                platform.x - camera.x + platform.width / 2,
                platform.y - camera.y + platform.height / 2
            );
            ctx.scale(1 + squishAmount * 0.2, 1 - squishAmount * 0.3);
            ctx.translate(
                -(platform.x - camera.x + platform.width / 2),
                -(platform.y - camera.y + platform.height / 2)
            );
        }

        // Hide respawning platforms
        if (platform.type === 'falling' && platform.state === 'respawning') {
            ctx.globalAlpha = 0.3 + (platform.fallTimer / platform.respawnTime) * 0.7;
        }

        // Island platform special rendering (floating islands with grass)
        if (platform.type === 'island') {
            const px = platform.x - camera.x + offsetX;
            const py = platform.y - camera.y + offsetY;
            const grassHeight = 12;

            // Brown dirt/earth base
            ctx.fillStyle = '#6D4C41';
            ctx.fillRect(px, py, platform.width, platform.height);

            // Add darker dirt shading for depth
            ctx.fillStyle = '#5D4037';
            ctx.fillRect(px, py + grassHeight, platform.width, platform.height - grassHeight);

            // Green grass top layer
            ctx.fillStyle = '#7CB342';
            ctx.fillRect(px, py, platform.width, grassHeight);

            // Lighter grass highlights
            ctx.fillStyle = '#9CCC65';
            ctx.fillRect(px, py, platform.width, grassHeight * 0.4);

            // Grass tufts on top
            ctx.fillStyle = '#8BC34A';
            for (let i = 0; i < platform.width; i += 15) {
                ctx.fillRect(px + i, py - 3, 8, 5);
            }

            // Dark outline
            ctx.strokeStyle = '#4E342E';
            ctx.lineWidth = 2;
            ctx.strokeRect(px, py, platform.width, platform.height);

            // Decorative tree on island
            if (platform.hasTree) {
                const treeX = px + platform.width / 2 - 15;
                const treeY = py - 45;

                // Tree trunk
                ctx.fillStyle = '#6D4C41';
                ctx.fillRect(treeX + 10, treeY + 20, 10, 25);

                // Tree foliage (3 layers)
                ctx.fillStyle = '#4CAF50';
                ctx.beginPath();
                ctx.arc(treeX + 15, treeY + 15, 18, 0, Math.PI * 2);
                ctx.fill();

                ctx.fillStyle = '#66BB6A';
                ctx.beginPath();
                ctx.arc(treeX + 15, treeY + 10, 15, 0, Math.PI * 2);
                ctx.fill();

                ctx.fillStyle = '#81C784';
                ctx.beginPath();
                ctx.arc(treeX + 15, treeY + 5, 12, 0, Math.PI * 2);
                ctx.fill();
            }

            // White pillar/column on island
            if (platform.hasPillar) {
                const pillarX = px + platform.width - 35;
                const pillarY = py - 60;

                // Pillar shaft
                ctx.fillStyle = '#FFFFFF';
                ctx.fillRect(pillarX, pillarY, 25, 60);

                // Pillar capital (top)
                ctx.fillRect(pillarX - 3, pillarY, 31, 8);

                // Pillar base
                ctx.fillRect(pillarX - 3, pillarY + 52, 31, 8);

                // Shading
                ctx.fillStyle = '#E0E0E0';
                ctx.fillRect(pillarX + 20, pillarY + 8, 5, 44);

                // Outline
                ctx.strokeStyle = '#BDBDBD';
                ctx.lineWidth = 1.5;
                ctx.strokeRect(pillarX, pillarY, 25, 60);
            }

            // Waterfall effect (falling from left side)
            if (platform.width > 150 && platform.y < 300) {
                const waterfallX = px + 10;
                const waterfallY = py + platform.height;

                ctx.fillStyle = 'rgba(135, 206, 250, 0.5)';
                ctx.fillRect(waterfallX, waterfallY, 8, 80);

                ctx.fillStyle = 'rgba(173, 216, 230, 0.3)';
                ctx.fillRect(waterfallX + 2, waterfallY, 4, 80);

                // Water splash at bottom
                ctx.fillStyle = 'rgba(135, 206, 250, 0.3)';
                ctx.beginPath();
                ctx.arc(waterfallX + 4, waterfallY + 80, 12, 0, Math.PI * 2);
                ctx.fill();
            }

            // Hanging vines (from right side)
            if (platform.width > 140 && Math.random() > 0.5 || platform.hasTree) {
                const vineX = px + platform.width - 20;
                const vineY = py + platform.height;

                ctx.strokeStyle = '#558B2F';
                ctx.lineWidth = 2;
                ctx.beginPath();
                ctx.moveTo(vineX, vineY);
                ctx.lineTo(vineX + 5, vineY + 30);
                ctx.lineTo(vineX - 5, vineY + 50);
                ctx.stroke();

                // Vine leaves
                ctx.fillStyle = '#7CB342';
                ctx.beginPath();
                ctx.arc(vineX + 5, vineY + 35, 4, 0, Math.PI * 2);
                ctx.fill();
                ctx.beginPath();
                ctx.arc(vineX - 5, vineY + 55, 4, 0, Math.PI * 2);
                ctx.fill();
            }
        }
        // Rainbow platform special rendering
        else if (platform.type === 'rainbow') {
            platform.rainbowOffset = (platform.rainbowOffset || 0) + 0.02;
            const gradient = ctx.createLinearGradient(
                platform.x - camera.x,
                platform.y - camera.y,
                platform.x - camera.x + platform.width,
                platform.y - camera.y
            );

            const offset = platform.rainbowOffset;
            gradient.addColorStop(0, `hsl(${(offset * 60) % 360}, 100%, 60%)`);
            gradient.addColorStop(0.33, `hsl(${(offset * 60 + 120) % 360}, 100%, 60%)`);
            gradient.addColorStop(0.66, `hsl(${(offset * 60 + 240) % 360}, 100%, 60%)`);
            gradient.addColorStop(1, `hsl(${(offset * 60) % 360}, 100%, 60%)`);

            ctx.fillStyle = gradient;

            ctx.fillRect(
                platform.x - camera.x + offsetX,
                platform.y - camera.y + offsetY,
                platform.width,
                platform.height
            );

            // Platform border
            ctx.strokeStyle = '#FFD700';
            ctx.lineWidth = 3;
            ctx.strokeRect(
                platform.x - camera.x + offsetX,
                platform.y - camera.y + offsetY,
                platform.width,
                platform.height
            );
        }
        // Standard platform rendering
        else {
            ctx.fillStyle = platform.color;

            ctx.fillRect(
                platform.x - camera.x + offsetX,
                platform.y - camera.y + offsetY,
                platform.width,
                platform.height
            );

            // Platform border
            ctx.strokeStyle = '#689F38';
            ctx.lineWidth = 3;
            ctx.strokeRect(
                platform.x - camera.x + offsetX,
                platform.y - camera.y + offsetY,
                platform.width,
                platform.height
            );
        }

        // Type indicators
        if (platform.type === 'bouncy') {
            ctx.fillStyle = 'rgba(255, 255, 255, 0.5)';
            ctx.font = 'bold 16px Arial';
            ctx.textAlign = 'center';
            ctx.fillText('↑', platform.x - camera.x + platform.width / 2 + offsetX, platform.y - camera.y + 15 + offsetY);
        }

        ctx.restore();
    }

    // Draw moving platforms (Phase 5)
    for (const platform of movingPlatforms) {
        ctx.fillStyle = platform.color;
        ctx.fillRect(
            platform.x - camera.x,
            platform.y - camera.y,
            platform.width,
            platform.height
        );

        // Platform border
        ctx.strokeStyle = '#558B2F';
        ctx.lineWidth = 2;
        ctx.strokeRect(
            platform.x - camera.x,
            platform.y - camera.y,
            platform.width,
            platform.height
        );

        // Type indicator (for debugging)
        ctx.fillStyle = 'rgba(0, 0, 0, 0.3)';
        ctx.font = '10px Arial';
        ctx.textAlign = 'center';
        if (platform.type === 'horizontal') ctx.fillText('↔', platform.x - camera.x + platform.width / 2, platform.y - camera.y + 14);
        if (platform.type === 'vertical') ctx.fillText('↕', platform.x - camera.x + platform.width / 2, platform.y - camera.y + 14);
        if (platform.type === 'circular') ctx.fillText('⭯', platform.x - camera.x + platform.width / 2, platform.y - camera.y + 14);
    }

    // Draw slopes (Phase 5)
    for (const slope of slopes) {
        ctx.save();
        ctx.fillStyle = slope.color;

        // Draw slope as a polygon (trapezoid/triangle)
        ctx.beginPath();
        ctx.moveTo(slope.x - camera.x, slope.y - camera.y); // Top-left
        ctx.lineTo(slope.x + slope.width - camera.x, slope.endY - camera.y); // Top-right (angled)
        ctx.lineTo(slope.x + slope.width - camera.x, slope.y + slope.height - camera.y); // Bottom-right
        ctx.lineTo(slope.x - camera.x, slope.y + slope.height - camera.y); // Bottom-left
        ctx.closePath();
        ctx.fill();

        // Border
        ctx.strokeStyle = '#689F38';
        ctx.lineWidth = 3;
        ctx.stroke();

        // Angle indicator
        ctx.fillStyle = 'rgba(0, 0, 0, 0.3)';
        ctx.font = '12px Arial';
        ctx.textAlign = 'center';
        const angleSymbol = slope.direction === 'up' ? '⬈' : '⬊';
        ctx.fillText(angleSymbol, slope.x - camera.x + slope.width / 2, slope.y - camera.y + slope.height / 2);

        ctx.restore();
    }

    // Draw ladders (Phase 5)
    for (const ladder of ladders) {
        ctx.fillStyle = ladder.color;
        ctx.fillRect(
            ladder.x - camera.x,
            ladder.y - camera.y,
            ladder.width,
            ladder.height
        );

        // Ladder rungs (draw horizontal lines for visual effect)
        ctx.strokeStyle = '#6D4C41';
        ctx.lineWidth = 3;
        const rungSpacing = 20;
        for (let y = ladder.y; y < ladder.y + ladder.height; y += rungSpacing) {
            ctx.beginPath();
            ctx.moveTo(ladder.x - camera.x, y - camera.y);
            ctx.lineTo(ladder.x - camera.x + ladder.width, y - camera.y);
            ctx.stroke();
        }

        // Ladder side rails
        ctx.strokeStyle = '#8D6E63';
        ctx.lineWidth = 4;
        ctx.beginPath();
        ctx.moveTo(ladder.x - camera.x + 3, ladder.y - camera.y);
        ctx.lineTo(ladder.x - camera.x + 3, ladder.y - camera.y + ladder.height);
        ctx.stroke();
        ctx.beginPath();
        ctx.moveTo(ladder.x - camera.x + ladder.width - 3, ladder.y - camera.y);
        ctx.lineTo(ladder.x - camera.x + ladder.width - 3, ladder.y - camera.y + ladder.height);
        ctx.stroke();
    }

    // Draw checkpoints (dog beds) - render before collectibles
    for (const checkpoint of checkpoints) {
        if (bedSprite && bedSprite.complete) {
            ctx.save();

            // Glow effect if activated
            if (checkpoint.activated) {
                ctx.shadowColor = '#FFD700';
                ctx.shadowBlur = 20;
            }

            ctx.drawImage(
                bedSprite,
                checkpoint.x - camera.x,
                checkpoint.y - camera.y,
                checkpoint.width,
                checkpoint.height
            );

            ctx.restore();
        }
    }

    // Draw obstacles (fire hydrants)
    for (const obstacle of obstacles) {
        if (obstacle.type === 'hydrant' && hydrantSprite && hydrantSprite.complete) {
            ctx.drawImage(
                hydrantSprite,
                obstacle.x - camera.x,
                obstacle.y - camera.y,
                obstacle.width,
                obstacle.height
            );
        }
    }

    // Draw collectibles (bones, balls, frisbees, water, food)
    for (const collectible of collectibles) {
        if (!collectible.collected) {
            let sprite = null;

            // Select sprite based on type
            if (collectible.type === 'bone') {
                sprite = boneSprites[collectible.boneVariant];
            } else if (collectible.type === 'ball') {
                sprite = ballSprite;
            } else if (collectible.type === 'frisbee') {
                sprite = frisbeeSprite;
            } else if (collectible.type === 'water') {
                sprite = waterBowlSprite;
            } else if (collectible.type === 'food') {
                sprite = foodBowlSprite;
            } else if (collectible.type === 'apple') {
                // Draw apple with canvas (red circle with leaf)
                ctx.save();
                const appleX = collectible.x - camera.x + collectible.width / 2;
                const appleY = collectible.y - camera.y + collectible.height / 2;

                // Red apple body
                ctx.fillStyle = '#FF3B30';
                ctx.beginPath();
                ctx.arc(appleX, appleY, collectible.width / 2, 0, Math.PI * 2);
                ctx.fill();

                // Highlight
                ctx.fillStyle = 'rgba(255, 255, 255, 0.3)';
                ctx.beginPath();
                ctx.arc(appleX - 3, appleY - 3, collectible.width / 4, 0, Math.PI * 2);
                ctx.fill();

                // Green leaf
                ctx.fillStyle = '#4CAF50';
                ctx.beginPath();
                ctx.ellipse(appleX + 5, appleY - 8, 4, 7, Math.PI / 4, 0, Math.PI * 2);
                ctx.fill();

                ctx.restore();
                continue;
            } else if (collectible.type === 'lightning') {
                // Draw lightning bolt (zigzag shape)
                ctx.save();
                const boltX = collectible.x - camera.x;
                const boltY = collectible.y - camera.y;
                const pulse = Math.sin((collectible.pulseTime || 0) * 3) * 0.2 + 1;

                ctx.globalAlpha = pulse;
                ctx.fillStyle = '#FFD700';
                ctx.strokeStyle = '#FFA500';
                ctx.lineWidth = 2;

                // Lightning bolt shape
                ctx.beginPath();
                ctx.moveTo(boltX + 15, boltY);
                ctx.lineTo(boltX + 10, boltY + 12);
                ctx.lineTo(boltX + 13, boltY + 12);
                ctx.lineTo(boltX + 8, boltY + 24);
                ctx.lineTo(boltX + 12, boltY + 14);
                ctx.lineTo(boltX + 10, boltY + 14);
                ctx.lineTo(boltX + 15, boltY);
                ctx.fill();
                ctx.stroke();

                ctx.restore();
                continue;
            }

            if (sprite && sprite.complete) {
                ctx.save();

                // Center position for rotation
                const centerX = collectible.x - camera.x + collectible.width / 2;
                const centerY = collectible.y - camera.y + collectible.height / 2;

                ctx.translate(centerX, centerY);

                // Rotation (frisbee spins faster, bowls don't rotate)
                if (collectible.type !== 'water' && collectible.type !== 'food') {
                    const rotationSpeed = collectible.type === 'frisbee' ? collectible.rotationSpeed * 2 : collectible.rotationSpeed;
                    ctx.rotate(collectible.floatTime * rotationSpeed);
                }

                // Draw sprite centered
                ctx.drawImage(
                    sprite,
                    -collectible.width / 2,
                    -collectible.height / 2,
                    collectible.width,
                    collectible.height
                );

                ctx.restore();
            }
        } else if (collectible.respawnTimer !== undefined && collectible.type === 'lightning') {
            // Draw respawning lightning bolt (ghost + timer)
            ctx.save();
            const boltX = collectible.x - camera.x;
            const boltY = collectible.y - camera.y;

            // Ghost lightning bolt (transparent)
            const respawnProgress = collectible.respawnTimer / collectible.respawnDuration;
            const ghostAlpha = 0.15 + (1 - respawnProgress) * 0.25; // Fades in as respawn approaches

            ctx.globalAlpha = ghostAlpha;
            ctx.fillStyle = '#FFD700';
            ctx.strokeStyle = '#FFA500';
            ctx.lineWidth = 2;

            // Lightning bolt shape
            ctx.beginPath();
            ctx.moveTo(boltX + 15, boltY);
            ctx.lineTo(boltX + 10, boltY + 12);
            ctx.lineTo(boltX + 13, boltY + 12);
            ctx.lineTo(boltX + 8, boltY + 24);
            ctx.lineTo(boltX + 12, boltY + 14);
            ctx.lineTo(boltX + 10, boltY + 14);
            ctx.lineTo(boltX + 15, boltY);
            ctx.fill();
            ctx.stroke();

            ctx.globalAlpha = 1.0;

            // Respawn timer text
            ctx.fillStyle = 'rgba(255, 255, 255, 0.9)';
            ctx.strokeStyle = 'rgba(0, 0, 0, 0.8)';
            ctx.lineWidth = 3;
            ctx.font = 'bold 14px Arial';
            ctx.textAlign = 'center';

            const timeText = Math.ceil(collectible.respawnTimer) + 's';
            ctx.strokeText(timeText, boltX + 12, boltY + 38);
            ctx.fillText(timeText, boltX + 12, boltY + 38);

            // Circular progress indicator
            const radius = 18;
            const centerX = boltX + 12;
            const centerY = boltY + 12;
            const startAngle = -Math.PI / 2;
            const endAngle = startAngle + (1 - respawnProgress) * Math.PI * 2;

            ctx.strokeStyle = 'rgba(100, 200, 255, 0.6)';
            ctx.lineWidth = 3;
            ctx.beginPath();
            ctx.arc(centerX, centerY, radius, startAngle, endAngle);
            ctx.stroke();

            ctx.restore();
        }
    }

    // Draw water zones (Phase 5)
    for (const water of waterZones) {
        ctx.save();
        ctx.fillStyle = water.color;
        ctx.fillRect(
            water.x - camera.x,
            water.y - camera.y,
            water.width,
            water.height
        );

        // Water surface animation (waves)
        ctx.strokeStyle = 'rgba(255, 255, 255, 0.3)';
        ctx.lineWidth = 2;
        const waveOffset = Date.now() / 500;
        ctx.beginPath();
        for (let x = 0; x < water.width; x += 10) {
            const y = Math.sin((x + waveOffset * 50) / 20) * 3;
            if (x === 0) {
                ctx.moveTo(water.x - camera.x + x, water.y - camera.y + y);
            } else {
                ctx.lineTo(water.x - camera.x + x, water.y - camera.y + y);
            }
        }
        ctx.stroke();

        ctx.restore();
    }

    // Draw interactive elements (Phase 5)
    for (const element of interactiveElements) {
        ctx.save();

        if (element.type === 'switch' || element.type === 'lever') {
            // Draw switch/lever
            ctx.fillStyle = element.activated ? '#4CAF50' : element.color;
            ctx.fillRect(
                element.x - camera.x,
                element.y - camera.y,
                element.width,
                element.height
            );

            // Border
            ctx.strokeStyle = element.activated ? '#2E7D32' : '#C62828';
            ctx.lineWidth = 3;
            ctx.strokeRect(
                element.x - camera.x,
                element.y - camera.y,
                element.width,
                element.height
            );

            // Indicator
            ctx.fillStyle = 'white';
            ctx.font = 'bold 12px Arial';
            ctx.textAlign = 'center';
            ctx.fillText(element.activated ? '✓' : '!',
                element.x - camera.x + element.width / 2,
                element.y - camera.y + element.height / 2 + 4
            );
        }

        if (element.type === 'door') {
            if (!element.open) {
                // Draw closed door
                ctx.fillStyle = element.color;
                ctx.fillRect(
                    element.x - camera.x,
                    element.y - camera.y,
                    element.width,
                    element.height
                );

                // Door panels
                ctx.strokeStyle = '#5D4037';
                ctx.lineWidth = 2;
                ctx.strokeRect(
                    element.x - camera.x + 5,
                    element.y - camera.y + 10,
                    element.width - 10,
                    element.height / 2 - 15
                );
                ctx.strokeRect(
                    element.x - camera.x + 5,
                    element.y - camera.y + element.height / 2 + 5,
                    element.width - 10,
                    element.height / 2 - 15
                );

                // Lock symbol if requires key
                if (element.requiresKey) {
                    ctx.fillStyle = '#FFD700';
                    ctx.font = 'bold 20px Arial';
                    ctx.textAlign = 'center';
                    ctx.fillText('🔒',
                        element.x - camera.x + element.width / 2,
                        element.y - camera.y + element.height / 2 + 7
                    );
                }
            }
        }

        if (element.type === 'bridge') {
            // Draw bridge
            const currentWidth = element.extended ? element.width : element.retractedWidth;
            ctx.fillStyle = element.color;
            ctx.fillRect(
                element.x - camera.x,
                element.y - camera.y,
                currentWidth,
                element.height
            );

            // Bridge planks
            ctx.strokeStyle = '#6D4C41';
            ctx.lineWidth = 2;
            for (let x = 0; x < currentWidth; x += 20) {
                ctx.beginPath();
                ctx.moveTo(element.x - camera.x + x, element.y - camera.y);
                ctx.lineTo(element.x - camera.x + x, element.y - camera.y + element.height);
                ctx.stroke();
            }
        }

        ctx.restore();
    }

    // Draw trees (foreground decorations)
    for (const tree of trees) {
        const treeSprite = treeSprites[tree.variant];
        if (treeSprite && treeSprite.complete) {
            ctx.drawImage(
                treeSprite,
                tree.x - camera.x,
                tree.y - camera.y,
                tree.width,
                tree.height
            );
        }
    }

    // Draw NPCs (Phase 3)
    for (const npc of npcs) {
        if (npc.sprite && npc.sprite.complete) {
            ctx.save();

            // Bark animation (scale pulse when greeted)
            if (npc.greeted && npc.barkTime < 1) {
                const scale = 1 + Math.sin(npc.barkTime * Math.PI * 3) * 0.15;
                const centerX = npc.x - camera.x + npc.width / 2;
                const centerY = npc.y - camera.y + npc.height / 2;
                ctx.translate(centerX, centerY);
                ctx.scale(scale, scale);
                ctx.translate(-centerX, -centerY);
            }

            // Flip sprite based on facing direction
            if (!npc.facingRight) {
                ctx.scale(-1, 1);
                ctx.drawImage(
                    npc.sprite,
                    -(npc.x - camera.x) - npc.width,
                    npc.y - camera.y,
                    npc.width,
                    npc.height
                );
            } else {
                ctx.drawImage(
                    npc.sprite,
                    npc.x - camera.x,
                    npc.y - camera.y,
                    npc.width,
                    npc.height
                );
            }

            ctx.restore();

            // Name tag (appears when nearby)
            const distanceToPlayer = Math.abs(player.x - npc.x);
            if (distanceToPlayer < 120 || npc.greeted) {
                ctx.save();
                ctx.font = '12px Arial';
                ctx.fillStyle = 'rgba(0, 0, 0, 0.7)';
                ctx.textAlign = 'center';
                const textWidth = ctx.measureText(npc.name).width + 8;
                ctx.fillRect(
                    npc.x - camera.x + npc.width / 2 - textWidth / 2,
                    npc.y - camera.y - 20,
                    textWidth,
                    16
                );
                ctx.fillStyle = npc.greeted ? '#FFD700' : 'white';
                ctx.fillText(npc.name, npc.x - camera.x + npc.width / 2, npc.y - camera.y - 8);
                ctx.restore();
            }
        }
    }

    // Draw squirrel (Phase 3)
    if (squirrel && !squirrel.caught) {
        ctx.save();

        // Draw simple squirrel (brown circle with tail)
        ctx.fillStyle = '#8B4513';
        ctx.beginPath();
        ctx.arc(
            squirrel.x - camera.x + squirrel.width / 2,
            squirrel.y - camera.y + squirrel.height / 2,
            squirrel.width / 2,
            0,
            Math.PI * 2
        );
        ctx.fill();

        // Bushy tail
        ctx.fillStyle = '#A0522D';
        ctx.beginPath();
        ctx.arc(
            squirrel.x - camera.x + squirrel.width,
            squirrel.y - camera.y + squirrel.height / 2,
            squirrel.width / 1.5,
            0,
            Math.PI * 2
        );
        ctx.fill();

        // Eyes
        ctx.fillStyle = 'black';
        ctx.beginPath();
        ctx.arc(
            squirrel.x - camera.x + squirrel.width / 2 - 3,
            squirrel.y - camera.y + squirrel.height / 2 - 3,
            2,
            0,
            Math.PI * 2
        );
        ctx.arc(
            squirrel.x - camera.x + squirrel.width / 2 + 3,
            squirrel.y - camera.y + squirrel.height / 2 - 3,
            2,
            0,
            Math.PI * 2
        );
        ctx.fill();

        ctx.restore();
    }

    // Draw particles (hearts, stars, zzz, sad)
    for (const particle of particles) {
        ctx.save();

        // Fade out over lifetime
        const alpha = 1 - (particle.elapsed / particle.lifetime);
        ctx.globalAlpha = alpha;

        ctx.font = '20px Arial';
        ctx.textAlign = 'center';

        if (particle.type === 'heart') {
            ctx.fillText('❤️', particle.x - camera.x, particle.y - camera.y);
        } else if (particle.type === 'star') {
            ctx.fillText('⭐', particle.x - camera.x, particle.y - camera.y);
        } else if (particle.type === 'zzz') {
            ctx.font = '16px Arial';
            ctx.fillStyle = '#888';
            ctx.fillText('💤', particle.x - camera.x, particle.y - camera.y);
        } else if (particle.type === 'sad') {
            ctx.font = '16px Arial';
            ctx.fillStyle = '#888';
            ctx.fillText('😢', particle.x - camera.x, particle.y - camera.y);
        } else if (particle.type === 'lightning') {
            // Lightning bolt particle
            ctx.strokeStyle = `rgba(255, 255, 100, ${alpha})`;
            ctx.lineWidth = 2;
            ctx.beginPath();
            ctx.moveTo(particle.x - camera.x, particle.y - camera.y);
            ctx.lineTo(particle.x - camera.x + 5, particle.y - camera.y + 8);
            ctx.lineTo(particle.x - camera.x + 2, particle.y - camera.y + 8);
            ctx.lineTo(particle.x - camera.x + 7, particle.y - camera.y + 16);
            ctx.stroke();

            // Glow effect
            ctx.fillStyle = `rgba(100, 200, 255, ${alpha * 0.5})`;
            ctx.fillRect(particle.x - camera.x - 2, particle.y - camera.y - 2, 11, 20);
        } else if (particle.type === 'water') {
            // Water bubble
            ctx.fillStyle = `rgba(150, 200, 255, ${alpha})`;
            ctx.strokeStyle = `rgba(200, 230, 255, ${alpha})`;
            ctx.lineWidth = 2;
            ctx.beginPath();
            ctx.arc(particle.x - camera.x, particle.y - camera.y, 5, 0, Math.PI * 2);
            ctx.fill();
            ctx.stroke();
        } else if (particle.type === 'rainbow') {
            // Rainbow sparkle
            const hue = (particle.elapsed * 360) % 360;
            ctx.fillStyle = `hsla(${hue}, 100%, 60%, ${alpha})`;
            ctx.font = '20px Arial';
            ctx.fillText('✨', particle.x - camera.x, particle.y - camera.y);
        }

        ctx.restore();
    }

    // Draw player
    const playerSprite = dogSprites[player.spriteIndex];

    if (playerSprite && playerSprite.complete) {
        ctx.save();

        // Speed powerup visual effect - electric glow!
        if (powerup.active && powerup.type === 'speed') {
            // Speed lines when moving
            if (Math.abs(player.velocityX) > 100) {
                const lineDirection = player.facingRight ? -1 : 1;
                ctx.strokeStyle = 'rgba(100, 200, 255, 0.6)';
                ctx.lineWidth = 3;

                for (let i = 0; i < 5; i++) {
                    const lineX = player.x - camera.x + (lineDirection * (20 + i * 15));
                    const lineY = player.y - camera.y + 15 + i * 8;
                    const lineLength = 30 - i * 4;

                    ctx.beginPath();
                    ctx.moveTo(lineX, lineY);
                    ctx.lineTo(lineX + lineDirection * lineLength, lineY);
                    ctx.stroke();
                }
            }

            // Lightning glow around player
            const glowRadius = 40 + Math.sin(Date.now() / 100) * 8;
            const gradient = ctx.createRadialGradient(
                player.x - camera.x + player.width / 2,
                player.y - camera.y + player.height / 2,
                0,
                player.x - camera.x + player.width / 2,
                player.y - camera.y + player.height / 2,
                glowRadius
            );
            gradient.addColorStop(0, 'rgba(255, 255, 100, 0.4)');
            gradient.addColorStop(0.5, 'rgba(100, 200, 255, 0.3)');
            gradient.addColorStop(1, 'rgba(100, 200, 255, 0)');

            ctx.fillStyle = gradient;
            ctx.fillRect(
                player.x - camera.x - glowRadius,
                player.y - camera.y - glowRadius,
                player.width + glowRadius * 2,
                player.height + glowRadius * 2
            );

            // Spawn lightning particles randomly
            if (Math.random() < 0.3) {
                particles.push({
                    x: player.x + Math.random() * player.width,
                    y: player.y + Math.random() * player.height,
                    velocityX: (Math.random() - 0.5) * 100,
                    velocityY: (Math.random() - 0.5) * 100,
                    lifetime: 0.5,
                    elapsed: 0,
                    type: 'lightning'
                });
            }
        }

        // Low energy visual effect (dim/desaturate)
        const energyPercent = stats.energy / stats.maxEnergy;
        if (energyPercent < 0.3 && !powerup.active) {
            ctx.globalAlpha = 0.7 + (energyPercent / 0.3) * 0.3; // Dim when low energy
        }

        // Low happiness visual effect (pulse when very low)
        const happinessPercent = stats.happiness / stats.maxHappiness;
        if (happinessPercent < 0.2) {
            const pulseScale = 1 + Math.sin(Date.now() / 200) * 0.05;
            const centerX = player.x - camera.x + player.width / 2;
            const centerY = player.y - camera.y + player.height / 2;
            ctx.translate(centerX, centerY);
            ctx.scale(pulseScale, pulseScale);
            ctx.translate(-centerX, -centerY);
        }

        if (!player.facingRight) {
            // Flip sprite horizontally
            ctx.scale(-1, 1);
            ctx.drawImage(
                playerSprite,
                -(player.x - camera.x) - player.width,
                player.y - camera.y,
                player.width,
                player.height
            );
        } else {
            ctx.drawImage(
                playerSprite,
                player.x - camera.x,
                player.y - camera.y,
                player.width,
                player.height
            );
        }

        ctx.restore();

        // Low stats indicator particles
        if (energyPercent < 0.2 && Math.random() < 0.05) {
            particles.push({
                x: player.x + player.width / 2,
                y: player.y + player.height / 2,
                velocityY: -30,
                lifetime: 1,
                elapsed: 0,
                type: 'zzz' // Sleepy indicator
            });
        }

        if (happinessPercent < 0.2 && Math.random() < 0.03) {
            particles.push({
                x: player.x + player.width / 2,
                y: player.y + 10,
                velocityY: -20,
                lifetime: 1.5,
                elapsed: 0,
                type: 'sad' // Sad indicator
            });
        }
    }

    // Draw interaction hints (Phase 5)
    ctx.save();

    // Check if near ladder
    for (const ladder of ladders) {
        if (checkCollision(player, ladder) && !player.climbing) {
            ctx.fillStyle = 'rgba(0, 0, 0, 0.7)';
            ctx.fillRect(player.x - camera.x - 10, player.y - camera.y - 40, 100, 30);
            ctx.fillStyle = '#FFD700';
            ctx.font = 'bold 14px Arial';
            ctx.textAlign = 'center';
            ctx.fillText('Press UP', player.x - camera.x + 40, player.y - camera.y - 20);
        }
    }

    // Check if near switch/lever
    for (const element of interactiveElements) {
        if ((element.type === 'switch' || element.type === 'lever') && !element.activated) {
            if (checkCollision(player, element)) {
                ctx.fillStyle = 'rgba(0, 0, 0, 0.7)';
                ctx.fillRect(element.x - camera.x - 10, element.y - camera.y - 40, 100, 30);
                ctx.fillStyle = '#4CAF50';
                ctx.font = 'bold 14px Arial';
                ctx.textAlign = 'center';
                ctx.fillText('Press UP', element.x - camera.x + 40, element.y - camera.y - 20);
            }
        }
    }

    ctx.restore();

    // Draw foreground parallax layers (Phase 5 - flowers in front of player)
    for (const element of foregroundLayers) {
        if (element.type === 'flower') {
            const flowerSprite = flowerSprites[element.variant];
            if (flowerSprite && flowerSprite.complete) {
                const parallaxX = element.x - (camera.x * element.parallaxSpeed);

                ctx.save();
                ctx.globalAlpha = 0.7; // Semi-transparent for foreground
                ctx.drawImage(
                    flowerSprite,
                    parallaxX,
                    element.y - camera.y,
                    element.size,
                    element.size
                );
                ctx.restore();
            }
        }
    }

    // DEBUG: Draw player hitbox (uncomment to debug)
    // ctx.strokeStyle = 'red';
    // ctx.lineWidth = 2;
    // ctx.strokeRect(player.x - camera.x, player.y - camera.y, player.width, player.height);

    // ===== PHASE 4 UI RENDERING =====

    // Draw Happiness Meter (top-left)
    if (happyStatSprite && happyStatSprite.complete) {
        const uiX = 20;
        const uiY = 20;
        const iconSize = 40;
        const barWidth = 200;
        const barHeight = 20;

        // Icon
        ctx.drawImage(happyStatSprite, uiX, uiY, iconSize, iconSize);

        // Background bar
        ctx.fillStyle = 'rgba(0, 0, 0, 0.5)';
        ctx.fillRect(uiX + iconSize + 10, uiY + 10, barWidth, barHeight);

        // Happiness bar
        const happinessPercent = stats.happiness / stats.maxHappiness;
        const happinessColor = happinessPercent > 0.6 ? '#4CAF50' : happinessPercent > 0.3 ? '#FFC107' : '#F44336';
        ctx.fillStyle = happinessColor;
        ctx.fillRect(uiX + iconSize + 10, uiY + 10, barWidth * happinessPercent, barHeight);

        // Border
        ctx.strokeStyle = 'white';
        ctx.lineWidth = 2;
        ctx.strokeRect(uiX + iconSize + 10, uiY + 10, barWidth, barHeight);

        // Text
        ctx.fillStyle = 'white';
        ctx.font = 'bold 14px Arial';
        ctx.fillText(`${Math.floor(stats.happiness)}/${stats.maxHappiness}`, uiX + iconSize + 15, uiY + 24);
    }

    // Draw Energy Meter (below happiness)
    if (energyStatSprite && energyStatSprite.complete) {
        const uiX = 20;
        const uiY = 70;
        const iconSize = 40;
        const barWidth = 200;
        const barHeight = 20;

        // Icon
        ctx.drawImage(energyStatSprite, uiX, uiY, iconSize, iconSize);

        // Background bar
        ctx.fillStyle = 'rgba(0, 0, 0, 0.5)';
        ctx.fillRect(uiX + iconSize + 10, uiY + 10, barWidth, barHeight);

        // Energy bar
        const energyPercent = stats.energy / stats.maxEnergy;
        const energyColor = energyPercent > 0.6 ? '#2196F3' : energyPercent > 0.3 ? '#FF9800' : '#F44336';
        ctx.fillStyle = energyColor;
        ctx.fillRect(uiX + iconSize + 10, uiY + 10, barWidth * energyPercent, barHeight);

        // Border
        ctx.strokeStyle = 'white';
        ctx.lineWidth = 2;
        ctx.strokeRect(uiX + iconSize + 10, uiY + 10, barWidth, barHeight);

        // Text
        ctx.fillStyle = 'white';
        ctx.font = 'bold 14px Arial';
        ctx.fillText(`${Math.floor(stats.energy)}/${stats.maxEnergy}`, uiX + iconSize + 15, uiY + 24);

        // Low energy warning
        if (energyPercent < 0.2) {
            ctx.fillStyle = '#F44336';
            ctx.font = 'bold 12px Arial';
            ctx.fillText('LOW ENERGY!', uiX + iconSize + 15, uiY + 40);
        }
    }

    // Draw Level/XP (top-right)
    ctx.save();
    ctx.fillStyle = 'rgba(0, 0, 0, 0.7)';
    ctx.fillRect(CANVAS_WIDTH - 150, 20, 130, 60);

    ctx.fillStyle = 'white';
    ctx.font = 'bold 18px Arial';
    ctx.fillText(`Level ${stats.level}`, CANVAS_WIDTH - 140, 45);

    // XP bar
    const levelThresholds = [0, 100, 250, 450, 700];
    const currentThreshold = levelThresholds[stats.level - 1] || 0;
    const nextThreshold = levelThresholds[stats.level] || 1000;
    const xpInLevel = stats.xp - currentThreshold;
    const xpNeeded = nextThreshold - currentThreshold;
    const xpPercent = stats.level >= 5 ? 1 : xpInLevel / xpNeeded;

    ctx.fillStyle = 'rgba(0, 0, 0, 0.5)';
    ctx.fillRect(CANVAS_WIDTH - 140, 52, 110, 15);

    ctx.fillStyle = '#FFD700';
    ctx.fillRect(CANVAS_WIDTH - 140, 52, 110 * xpPercent, 15);

    ctx.strokeStyle = 'white';
    ctx.lineWidth = 2;
    ctx.strokeRect(CANVAS_WIDTH - 140, 52, 110, 15);

    ctx.fillStyle = 'white';
    ctx.font = '10px Arial';
    if (stats.level < 5) {
        ctx.fillText(`${xpInLevel}/${xpNeeded} XP`, CANVAS_WIDTH - 135, 63);
    } else {
        ctx.fillText('MAX LEVEL', CANVAS_WIDTH - 135, 63);
    }
    ctx.restore();

    // Draw Collection Tracker (toggle with C key)
    if (showCollectionTracker) {
        const trackerX = CANVAS_WIDTH / 2 - 200;
        const trackerY = CANVAS_HEIGHT - 180;
        const trackerWidth = 400;
        const trackerHeight = 160;

        // Background
        ctx.save();
        ctx.fillStyle = 'rgba(0, 0, 0, 0.85)';
        ctx.fillRect(trackerX, trackerY, trackerWidth, trackerHeight);
        ctx.strokeStyle = '#FFD700';
        ctx.lineWidth = 3;
        ctx.strokeRect(trackerX, trackerY, trackerWidth, trackerHeight);

        // Title
        ctx.fillStyle = '#FFD700';
        ctx.font = 'bold 20px Arial';
        ctx.fillText('Collection Progress', trackerX + 120, trackerY + 30);

        // Items
        ctx.font = '16px Arial';
        ctx.fillStyle = 'white';
        const lineHeight = 22;
        let yOffset = trackerY + 60;

        const collections = [
            { emoji: '🦴', name: 'Bones', current: collectionStats.bonesCollected, total: collectionStats.bonesTotal },
            { emoji: '⚾', name: 'Balls', current: collectionStats.ballsCollected, total: collectionStats.ballsTotal },
            { emoji: '🥏', name: 'Frisbees', current: collectionStats.frisbeesCollected, total: collectionStats.frisbeesTotal },
            { emoji: '🐶', name: 'NPCs Met', current: collectionStats.npcsGreeted, total: collectionStats.npcsTotal },
            { emoji: '🐿️', name: 'Squirrel', current: collectionStats.squirrelCaught ? 1 : 0, total: 1 }
        ];

        collections.forEach((item, index) => {
            const complete = item.current >= item.total;
            ctx.fillStyle = complete ? '#4CAF50' : 'white';
            ctx.fillText(`${item.emoji} ${item.name}: ${item.current}/${item.total}`, trackerX + 20, yOffset + index * lineHeight);

            if (complete) {
                ctx.fillText('✓', trackerX + 340, yOffset + index * lineHeight);
            }
        });

        // Footer
        ctx.fillStyle = '#888';
        ctx.font = '12px Arial';
        ctx.fillText('Press C to close', trackerX + 150, trackerY + trackerHeight - 10);
        ctx.restore();
    }

    // Draw Zone Name (Phase 5)
    ctx.save();
    const zoneName = zones[currentZone]?.name || 'Unknown';
    ctx.fillStyle = 'rgba(0, 0, 0, 0.5)';
    ctx.fillRect(10, CANVAS_HEIGHT - 40, 200, 30);
    ctx.fillStyle = '#FFD700';
    ctx.font = 'bold 16px Arial';
    ctx.textAlign = 'left';
    ctx.fillText(`📍 ${zoneName}`, 20, CANVAS_HEIGHT - 18);
    ctx.restore();

    // Draw Achievement Popup (compact, top-right corner)
    if (achievementPopup) {
        const popupWidth = 280;
        const popupHeight = 70;
        const popupX = CANVAS_WIDTH - popupWidth - 10;
        const popupY = 100;

        ctx.save();

        // Slide-in animation from right
        const slideProgress = Math.min(1, (3 - achievementPopup.timeLeft) / 0.5);
        const offsetX = (1 - slideProgress) * 300;

        // Background with transparency
        ctx.fillStyle = 'rgba(0, 0, 0, 0.85)';
        ctx.fillRect(popupX + offsetX, popupY, popupWidth, popupHeight);

        // Gold border
        ctx.strokeStyle = '#FFD700';
        ctx.lineWidth = 2;
        ctx.strokeRect(popupX + offsetX, popupY, popupWidth, popupHeight);

        // Achievement icon
        ctx.font = '28px Arial';
        ctx.fillText(achievementPopup.achievement.icon, popupX + offsetX + 15, popupY + 42);

        // Title and name combined
        ctx.fillStyle = '#FFD700';
        ctx.font = 'bold 14px Arial';
        ctx.fillText('🏆 ' + achievementPopup.achievement.name, popupX + offsetX + 55, popupY + 28);

        // Description
        ctx.fillStyle = '#CCC';
        ctx.font = '12px Arial';
        ctx.fillText(achievementPopup.achievement.desc, popupX + offsetX + 55, popupY + 48);

        ctx.restore();
    }

    // Draw Speed Powerup Indicator (top-left corner)
    if (powerup.active && powerup.type === 'speed') {
        const powerupWidth = 220;
        const powerupHeight = 60;
        const powerupX = 10;
        const powerupY = 80;

        ctx.save();

        // Pulsing background
        const pulseAlpha = 0.85 + Math.sin(Date.now() / 200) * 0.15;
        ctx.globalAlpha = pulseAlpha;

        // Background with electric gradient
        const gradient = ctx.createLinearGradient(powerupX, powerupY, powerupX + powerupWidth, powerupY);
        gradient.addColorStop(0, 'rgba(50, 50, 100, 0.9)');
        gradient.addColorStop(0.5, 'rgba(100, 150, 255, 0.9)');
        gradient.addColorStop(1, 'rgba(50, 50, 100, 0.9)');

        ctx.fillStyle = gradient;
        ctx.fillRect(powerupX, powerupY, powerupWidth, powerupHeight);

        // Border glow
        ctx.strokeStyle = 'rgba(255, 255, 100, 0.8)';
        ctx.lineWidth = 3;
        ctx.strokeRect(powerupX, powerupY, powerupWidth, powerupHeight);

        ctx.globalAlpha = 1.0;

        // Lightning icon
        ctx.font = '32px Arial';
        ctx.fillText('⚡', powerupX + 15, powerupY + 42);

        // Title
        ctx.fillStyle = '#FFD700';
        ctx.font = 'bold 16px Arial';
        ctx.textAlign = 'left';
        ctx.fillText('SPEED BOOST!', powerupX + 55, powerupY + 22);

        // Timer bar background
        const barX = powerupX + 55;
        const barY = powerupY + 30;
        const barWidth = 150;
        const barHeight = 18;

        ctx.fillStyle = 'rgba(0, 0, 0, 0.5)';
        ctx.fillRect(barX, barY, barWidth, barHeight);

        // Timer bar fill
        const fillWidth = (powerup.timeLeft / powerup.duration) * barWidth;
        const barGradient = ctx.createLinearGradient(barX, barY, barX + fillWidth, barY);
        barGradient.addColorStop(0, '#FFD700');
        barGradient.addColorStop(1, '#FFA500');

        ctx.fillStyle = barGradient;
        ctx.fillRect(barX, barY, fillWidth, barHeight);

        // Timer text
        ctx.fillStyle = '#FFF';
        ctx.font = 'bold 13px Arial';
        ctx.textAlign = 'center';
        ctx.fillText(`${Math.ceil(powerup.timeLeft)}s`, barX + barWidth / 2, barY + 13);

        ctx.restore();
    }
}

// ===== DEBUG TOGGLE FUNCTION =====
function toggleDebug() {
    const debugInfo = document.getElementById('debugInfo');
    if (debugInfo.style.display === 'block') {
        debugInfo.style.display = 'none';
    } else {
        const hasClass = document.documentElement.classList.contains('mobile-device');
        const controlsVisible = window.getComputedStyle(document.getElementById('leftBtn')).display !== 'none';

        const leftBtn = document.getElementById('leftBtn');
        const leftStyle = leftBtn ? window.getComputedStyle(leftBtn) : null;

        debugInfo.innerHTML = `
            <strong>Debug Info:</strong><br>
            Mobile Class: ${hasClass ? 'YES' : 'NO'}<br>
            Controls Visible: ${controlsVisible ? 'YES' : 'NO'}<br>
            Window Width: ${window.innerWidth}px<br>
            Window Height: ${window.innerHeight}px<br>
            Has Touch: ${'ontouchstart' in window ? 'YES' : 'NO'}<br>
            Touch Points: ${navigator.maxTouchPoints}<br>
            Left Btn Display: ${leftStyle ? leftStyle.display : 'N/A'}<br>
            Left Btn Position: ${leftStyle ? leftStyle.position : 'N/A'}<br>
            Left Btn Z-Index: ${leftStyle ? leftStyle.zIndex : 'N/A'}<br>
            Buttons Found: ${document.querySelectorAll('.controlBtn').length}<br>
            <button onclick="
                document.querySelectorAll('.controlBtn').forEach(el => {
                    el.style.display='block';
                    el.style.position='fixed';
                    el.style.zIndex='99999';
                    el.style.background='red';
                    el.style.border='5px solid yellow';
                    el.style.opacity='1';
                    el.style.visibility='visible';
                });
                alert('Controls forced visible with red/yellow styling');
            " style="margin-top:10px; padding:8px; background:#4CAF50; color:white; border:none; border-radius:5px; font-size:14px;">
                Force Show Controls
            </button>
        `;
        debugInfo.style.display = 'block';
    }
}

// Show debug button on mobile or when needed
window.addEventListener('DOMContentLoaded', () => {
    const debugBtn = document.getElementById('debugBtn');
    // Show debug button on mobile or narrow screens
    if (window.innerWidth <= 768 || 'ontouchstart' in window) {
        debugBtn.style.display = 'block';
    }
});

// ===== START LOADING =====
// ===== START LOADING =====
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', loadAssets);
} else {
    // DOM already loaded (SPA navigation)
    loadAssets();
}

// ===== PREVENT SCROLLING ON MOBILE =====
document.addEventListener('touchmove', (e) => {
    e.preventDefault();
}, { passive: false });

// ===== EXPOSE TO WINDOW FOR DEBUGGING =====
window.game = {
    player,
    camera,
    platforms,
    collectibles,
    reset: () => {
        score = 0;
        document.getElementById('score').textContent = 'Score: 0';
        player.x = 150;
        player.y = platforms[1].y - player.height - 2; // Spawn on starting platform
        player.velocityX = 0;
        player.velocityY = 0;
        collectibles.forEach(c => c.collected = false);

        // Reset checkpoints
        checkpoints.forEach((checkpoint, i) => {
            checkpoint.activated = (i === 0); // Only first is active
        });
        currentCheckpoint = 0;

        // Reset NPCs (Phase 3)
        npcs.forEach(npc => {
            npc.greeted = false;
            npc.barkTime = 0;
        });

        // Reset squirrel (Phase 3)
        if (squirrel) {
            squirrel.caught = false;
            squirrel.respawnTimer = 0;
            squirrel.x = 750;
            squirrel.y = platforms[5].y - 180;
            squirrel.state = 'idle';
            squirrel.chasePhase = 0;
        }

        // Clear particles
        particles.length = 0;

        console.log('🔄 Game reset! All collectibles, NPCs, and squirrel restored.');
    }
};

console.log('🎮 Debug: window.game.player, window.game.camera available');
console.log('🎮 Debug: window.game.reset() to restart');
