// Get the canvas element
const canvas = document.getElementById('pong');
const context = canvas.getContext('2d');

// Get the audio element
const hitSound = document.getElementById('hitSound');

// Game objects
const ball = {
    x: canvas.width / 2,
    y: canvas.height / 2,
    size: 10,
    speed: 5,
    velocityX: 5,
    velocityY: 5
};

const paddleHeight = 100;
const paddleWidth = 10;
const player1 = {
    y: canvas.height / 2 - paddleHeight / 2,
    score: 0
};
const player2 = {
    y: canvas.height / 2 - paddleHeight / 2,
    score: 0,
    speed: 5  // AI paddle speed
};

// Mouse position
let mouseY = canvas.height / 2;

// Event listener for mouse movement
canvas.addEventListener('mousemove', (e) => {
    const rect = canvas.getBoundingClientRect();
    mouseY = e.clientY - rect.top;
});

// Game functions
function moveAI() {
    // Calculate the target position for AI
    const targetY = ball.y - paddleHeight / 2;
    
    // Add some delay and imperfection to make AI beatable
    if (ball.velocityX > 0) {  // Only move when ball is coming towards AI
        // Add some randomness to make AI imperfect
        const reactionDelay = Math.random() * 0.3;
        
        // Move towards the ball with some delay
        if (player2.y + paddleHeight / 2 < targetY) {
            player2.y += player2.speed * (1 - reactionDelay);
        } else if (player2.y + paddleHeight / 2 > targetY) {
            player2.y -= player2.speed * (1 - reactionDelay);
        }
        
        // Keep AI paddle within bounds
        player2.y = Math.max(0, Math.min(canvas.height - paddleHeight, player2.y));
    }
}

function movePaddles() {
    // Player 1 follows mouse position
    const targetY = mouseY - paddleHeight / 2;
    player1.y += (targetY - player1.y) * 0.1;
    
    // Keep player 1 paddle within bounds
    player1.y = Math.max(0, Math.min(canvas.height - paddleHeight, player1.y));
    
    // Move AI paddle
    moveAI();
}

function moveBall() {
    ball.x += ball.velocityX;
    ball.y += ball.velocityY;

    // Collision with top and bottom walls
    if (ball.y + ball.size > canvas.height || ball.y - ball.size < 0) {
        ball.velocityY = -ball.velocityY;
    }

    // Collision with paddles
    if (ball.x - ball.size < paddleWidth && 
        ball.y > player1.y && 
        ball.y < player1.y + paddleHeight) {
        ball.velocityX = -ball.velocityX;
        ball.speed += 0.5;
        // Play sound with volume
        hitSound.volume = 1.0;
        hitSound.currentTime = 0;
        hitSound.play().catch(error => console.log("Sound play failed:", error));
    }
    if (ball.x + ball.size > canvas.width - paddleWidth && 
        ball.y > player2.y && 
        ball.y < player2.y + paddleHeight) {
        ball.velocityX = -ball.velocityX;
        ball.speed += 0.5;
        // Play sound with volume
        hitSound.volume = 1.0;
        hitSound.currentTime = 0;
        hitSound.play().catch(error => console.log("Sound play failed:", error));
    }

    // Score points
    if (ball.x - ball.size < 0) {
        player2.score++;
        resetBall();
    }
    if (ball.x + ball.size > canvas.width) {
        player1.score++;
        resetBall();
    }
}

function resetBall() {
    ball.x = canvas.width / 2;
    ball.y = canvas.height / 2;
    ball.speed = 5;
    ball.velocityX = -ball.velocityX;
    ball.velocityY = Math.random() * 10 - 5;
}

function draw() {
    // Clear canvas
    context.fillStyle = '#000';
    context.fillRect(0, 0, canvas.width, canvas.height);

    // Draw middle line
    context.strokeStyle = '#fff';
    context.beginPath();
    context.setLineDash([5, 15]);
    context.moveTo(canvas.width / 2, 0);
    context.lineTo(canvas.width / 2, canvas.height);
    context.stroke();
    context.setLineDash([]);

    // Draw paddles
    context.fillStyle = '#fff';
    context.fillRect(0, player1.y, paddleWidth, paddleHeight);
    context.fillRect(canvas.width - paddleWidth, player2.y, paddleWidth, paddleHeight);

    // Draw ball
    context.beginPath();
    context.arc(ball.x, ball.y, ball.size, 0, Math.PI * 2);
    context.fill();

    // Draw scores
    context.font = '32px Monospace';
    context.fillText(player1.score, canvas.width / 4, 50);
    context.fillText(player2.score, 3 * canvas.width / 4, 50);
}

// Game loop
function gameLoop() {
    movePaddles();
    moveBall();
    draw();
    requestAnimationFrame(gameLoop);
}

// Start the game
gameLoop();
