// Word list
const words = [
    'APPLE', 'BEACH', 'CLOUD', 'DANCE', 'EARTH',
    'FLAME', 'GRAPE', 'HAPPY', 'IVORY', 'JUMBO',
    'KNIFE', 'LEMON', 'MUSIC', 'NOBLE', 'OCEAN',
    'PIANO', 'QUEEN', 'RIVER', 'SMILE', 'TIGER'
];

// Game state
let currentWord = '';
let guessedLetters = new Set();
let remainingTries = 6;
let gameOver = false;

// DOM elements
const wordDisplay = document.getElementById('word-display');
const keyboard = document.getElementById('keyboard');
const messageElement = document.getElementById('message');
const triesLeftElement = document.getElementById('tries-left');

// Initialize game
function initGame() {
    currentWord = words[Math.floor(Math.random() * words.length)];
    guessedLetters = new Set();
    remainingTries = 6;
    gameOver = false;
    createKeyboard();
    updateDisplay();
    updateTriesLeft();
}

// Create keyboard
function createKeyboard() {
    const rows = [
        ['Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P'],
        ['A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L'],
        ['Z', 'X', 'C', 'V', 'B', 'N', 'M']
    ];

    keyboard.innerHTML = '';
    rows.forEach(row => {
        const rowDiv = document.createElement('div');
        rowDiv.className = 'keyboard-row';
        
        row.forEach(letter => {
            const button = document.createElement('button');
            button.className = 'key';
            button.textContent = letter;
            button.addEventListener('click', () => handleGuess(letter));
            rowDiv.appendChild(button);
        });
        
        keyboard.appendChild(rowDiv);
    });
}

// Update word display
function updateDisplay() {
    wordDisplay.innerHTML = '';
    
    for (let letter of currentWord) {
        const letterBox = document.createElement('div');
        letterBox.className = 'letter-box';
        
        if (guessedLetters.has(letter)) {
            letterBox.textContent = letter;
            letterBox.classList.add('correct');
        } else {
            letterBox.textContent = '_';
        }
        
        wordDisplay.appendChild(letterBox);
    }
}

// Handle guess
function handleGuess(letter) {
    if (gameOver || guessedLetters.has(letter)) return;

    guessedLetters.add(letter);
    const button = [...document.querySelectorAll('.key')]
        .find(btn => btn.textContent === letter);

    if (currentWord.includes(letter)) {
        button.classList.add('correct');
        messageElement.textContent = 'Good guess!';
    } else {
        button.classList.add('absent');
        remainingTries--;
        updateTriesLeft();
        messageElement.textContent = 'Try again!';
    }

    updateDisplay();
    checkGameEnd();
}

// Update tries left display
function updateTriesLeft() {
    triesLeftElement.textContent = `Tries left: ${remainingTries}`;
}

// Check if game is over
function checkGameEnd() {
    const allLettersGuessed = [...currentWord]
        .every(letter => guessedLetters.has(letter));

    if (allLettersGuessed) {
        messageElement.textContent = 'Congratulations! You won! 🎉';
        gameOver = true;
        addPlayAgainButton();
    } else if (remainingTries <= 0) {
        messageElement.textContent = `Game Over! The word was ${currentWord}`;
        gameOver = true;
        addPlayAgainButton();
    }
}

// Add play again button
function addPlayAgainButton() {
    const playAgainButton = document.createElement('button');
    playAgainButton.textContent = 'Play Again';
    playAgainButton.className = 'key';
    playAgainButton.style.marginTop = '20px';
    playAgainButton.addEventListener('click', initGame);
    messageElement.appendChild(playAgainButton);
}

// Handle keyboard input
document.addEventListener('keydown', (e) => {
    if (gameOver) return;
    
    const key = e.key.toUpperCase();
    if (/^[A-Z]$/.test(key)) {
        handleGuess(key);
    }
});

// Start the game
initGame();
